use std::ffi::CStr;
use std::mem::size_of;
use std::os::unix::prelude::*;

use input_event_codes::{ABS_CNT, BTN_MISC, KEY_CNT};
use libc::read;
use nix::errno::Errno;

use crate::event_codes::{AbsoluteAxis, KeyOrButton};
use crate::ioctl::{
	jsiocgaxes, jsiocgaxmap, jsiocgbtnmap, jsiocgbuttons, jsiocgcorr, jsiocgname, jsiocgversion, jsiocsaxmap,
	jsiocsbtnmap, jsiocscorr,
};
use crate::{Correction, Error, Event, Result};

pub fn get_axis_count(fd: RawFd) -> Result<u8> {
	let mut result = 0u8;
	unsafe { jsiocgaxes(fd, &mut result) }?;
	Ok(result)
}

pub fn get_axis_mapping(fd: RawFd) -> Result<Vec<AbsoluteAxis>> {
	let mut result = vec![AbsoluteAxis::default(); ABS_CNT as usize];
	unsafe { jsiocgaxmap(fd, result.as_mut_ptr() as *mut _) }?;
	Ok(result)
}

pub fn get_button_count(fd: RawFd) -> Result<u8> {
	let mut result = 0u8;
	unsafe { jsiocgbuttons(fd, &mut result) }?;
	Ok(result)
}

pub fn get_button_mapping(fd: RawFd) -> Result<Vec<KeyOrButton>> {
	let mut result = vec![KeyOrButton::default(); (KEY_CNT - BTN_MISC) as usize];
	unsafe { jsiocgbtnmap(fd, result.as_mut_ptr() as *mut _) }?;
	Ok(result)
}

pub fn get_correction_values(fd: RawFd) -> Result<Vec<Correction>> {
	let mut result = vec![Correction::default(); ABS_CNT as usize];
	unsafe { jsiocgcorr(fd, result.as_mut_ptr() as *mut _) }?;
	Ok(result)
}

pub fn get_driver_version(fd: RawFd) -> Result<u32> {
	let mut result = 0u32;
	unsafe { jsiocgversion(fd, &mut result) }?;
	Ok(result)
}

pub fn get_event(fd: RawFd) -> Result<Event> {
	let mut result = Event::default();
	if unsafe { read(fd, (&mut result as *mut _) as *mut _, size_of::<Event>()) } > 0 {
		Ok(result)
	} else {
		let errno = Errno::last();
		match errno {
			Errno::EAGAIN => Err(Error::QueueEmpty),
			_ => Err(Error::from(errno)),
		}
	}
}

/*pub fn get_event_legacy(fd: RawFd) -> Result<JS_DATA_TYPE> {
	unimplemented!()
}*/

pub fn get_identifier(fd: RawFd) -> Result<String> {
	// TODO: Maybe change to vector?
	let mut result = [0i8; 128];
	unsafe { jsiocgname(fd, &mut result) }?;
	Ok(unsafe { CStr::from_ptr(&result as *const _) }.to_str()?.to_owned())
}

pub fn set_axis_mapping(fd: RawFd, mapping: &[AbsoluteAxis; ABS_CNT as usize]) -> Result<()> {
	unsafe { jsiocsaxmap(fd, mapping as *const _ as *const _) }?;
	Ok(())
}

pub fn set_button_mapping(fd: RawFd, mapping: &[KeyOrButton; (KEY_CNT - BTN_MISC) as usize]) -> Result<()> {
	unsafe { jsiocsbtnmap(fd, mapping as *const _ as *const _) }?;
	Ok(())
}

pub fn set_correction_values(fd: RawFd, correction: &[Correction; ABS_CNT as usize]) -> Result<()> {
	unsafe { jsiocscorr(fd, correction as *const _ as *const _) }?;
	Ok(())
}
