use std::io::Error as IoError;
use std::str::Utf8Error;
use std::{error, fmt};

use nix::errno::Errno;

#[derive(Debug)]
pub enum Error {
	Io(IoError),
	QueueEmpty,
	String(Utf8Error),
	Sys(Errno),
}

impl error::Error for Error {
	fn description(&self) -> &str {
		match self {
			Self::Io(_) => "IO error",
			Self::QueueEmpty => "Queue empty",
			Self::String(_) => "String error",
			Self::Sys(ref errno) => errno.desc(),
		}
	}

	fn source(&self) -> Option<&(dyn error::Error + 'static)> {
		match self {
			Self::Io(ref error) => Some(error),
			Self::String(ref error) => Some(error),
			_ => None,
		}
	}
}

impl fmt::Display for Error {
	fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
		match self {
			Self::Io(ref error) => write!(f, "{}", error),
			Self::QueueEmpty => write!(f, "Queue empty"),
			Self::String(ref error) => write!(f, "{}", error),
			Self::Sys(ref errno) => write!(f, "{:?}: {}", errno, errno.desc()),
		}
	}
}

impl From<Errno> for Error {
	fn from(errno: Errno) -> Self {
		Error::Sys(errno)
	}
}

impl From<IoError> for Error {
	fn from(error: IoError) -> Self {
		Self::Io(error)
	}
}

impl From<Utf8Error> for Error {
	fn from(error: Utf8Error) -> Self {
		Error::String(error)
	}
}
