use std::fmt;
use std::mem::{transmute, zeroed};

use sys::js_corr;

use crate::CorrectionType;

#[derive(Clone, Copy)]
pub struct Correction {
	pub(crate) corr: js_corr,
}

impl Correction {
	pub const fn coefficient(&self) -> &[i32; 8] {
		&self.corr.coef
	}

	pub const fn new(coefficient: &[i32; 8], precision: i16, type_: CorrectionType) -> Self {
		Self {
			corr: js_corr {
				coef: *coefficient,
				prec: precision,
				type_: type_ as u16,
			},
		}
	}

	pub const fn precision(&self) -> i16 {
		self.corr.prec
	}

	pub fn type_(&self) -> CorrectionType {
		unsafe { transmute(self.corr.type_) }
	}
}

impl Default for Correction {
	fn default() -> Self {
		Self {
			corr: unsafe { zeroed() },
		}
	}
}

impl fmt::Debug for Correction {
	fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
		write!(
			f,
			"Correction {{ coefficient: {:?}, precision: {:?}, type: {:?} }}",
			self.coefficient(),
			self.precision(),
			self.type_()
		)
	}
}
