# Inter-Struct

[![GitHub Actions Workflow](https://github.com/nukesor/inter-struct/workflows/Test%20build/badge.svg)](https://github.com/Nukesor/inter-struct/actions)
[![Crates.io](https://img.shields.io/crates/v/inter-struct)](https://crates.io/crates/inter-struct)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Downloads](https://img.shields.io/github/downloads/nukesor/inter-struct/total.svg)](https://github.com/nukesor/inter-struct/releases)

A `#[derive()]` macro for various trait implementations between two structs.

Please read the **known caveats** section before using this crate!
It's not trivial to implement code for two structs in the same codebase.

Also note that this crate is in an early development phase.
The crate is already properly tested, but bugs might still be there.

## Features:

- [ ] PartialEq
- [x] Merge
- [x] MergeRef
- [x] Into - A standard `From/Into` impl between two structs.
- [x] IntoDefault - `From/Into`, but use `Default` on the target for unknown fields.
- [ ] Field attributes to customize the behavior of our traits.
    * [ ] `unchecked` Ignore any type checks done by inter-struct and let the compiler handle errors.
    * [ ] `rename` Similar to serde's rename, map a field to another named field.
    * [ ] `ignore` Ignore this field in the type generation.

## Merge

```rust,ignore
/// Merge another struct into Self whilst consuming it.
/// 
/// The other trait is named `StructMergeRef` and merges other structs by reference.
pub trait StructMerge<Src> {
    /// Merge the given struct into self.
    fn merge(&mut self, src: Src);
}
```

This following code is an example on how to use the `InterStruct` derive macro for implementing the `StructMerge` trait between two structs.

```rust,ignore
use inter_struct::prelude::*;

/// The target struct we'll merge into.
pub struct Target {
    pub normal: String,
    pub optional: String,
    /// This field won't be touched as the macro cannot find a
    /// respective `ignored` field in the `Source` struct.
    pub ignored: String,
}

/// A struct with both an identical and an optional field type.
/// Note that the path to `Target` must always be fully qualifying.
#[derive(InterStruct)]
#[merge("crate::Target")]
pub struct Source {
    pub normal: String,
    pub optional: Option<String>,
}

fn main() {
    let mut target = Target {
        normal: "target".to_string(),
        optional: "target".to_string(),
        ignored: "target".to_string(),
    };

    let source = Source {
        /// Has the same type as Target::normal
        normal: "source".to_string(),
        /// Wraps Target::optional in an Option
        optional: Some("source".to_string()),
    };

    // Merge the `Source` struct into target.
    target.merge(source);
    // You can also call this:
    // source.merge_into(target);
    assert_eq!(target.normal, "source".to_string());
    assert_eq!(target.optional, Some("source".to_string()));
    assert_eq!(target.ignored, "target".to_string());
}
```


## Into


This following code is an example on how to use the `InterStruct` derive macro for implementing `Into` between two structs.

```rust,ignore
use inter_struct::prelude::*;

/// The target struct we'll convert our `Source` struct into.
pub struct Target {
    pub normal: String,
    pub optional: String,
}

#[derive(InterStruct)]
// Note that the path to `Target` must always be fully qualifying.
#[into("crate::Target")]
pub struct Source {
    pub normal: String,
    pub optional: Option<String>,
    /// This field doesn't exist in the target, hence it'll be ignored.
    pub ignored: String,
}

fn main() {
    let source = Source {
        /// Has the same type as Target::normal
        normal: "source".to_string(),
        /// Wraps Target::optional in an Option
        optional: Some("source".to_string()),
        ignored: "source".to_string(),
    };

    // Merge the `Source` struct into target.
    let target: Target = source.into();
    assert_eq!(target.normal, "source".to_string());
    assert_eq!(target.optional, Some("source".to_string()));
}
```

## Known caveats

Inter-struct is designed to work in this environment:

- In the scope of a single crate. Cross-crate usage won't work.
- In the main `src` folder of the crate. Integration tests and examples aren't supported.

The main problems in this crate come from the fact that there's no official way to resolve modules or types in the the procedural macro stage.

Due to this limitation, inter-struct isn't capable of ensuring the equality of two types.
As a result, it might create false negative compile errors, even though the types might be compatible.
This might happen if, for instance, types are obscured via an alias or if a type can be automatically dereferenced into another type.

However, as we're creating safe and valid Rust code, the compiler will thrown an error if any type problems arise.


#### Not yet solved problems

These are problems that can probably be solved but they're non-trivial.

- [ ] Struct located at root of crate. E.g. `lib.rs`.
- [ ] Struct is located in integration tests.
- [ ] Struct in (potentially nested or alternating) `mod {}` block in file.
- [ ] The source root dir isn't `src`.
      We would have to check the environment and possibly parse the `Cargo.toml`.
- [ ] Different generic aliases that use different tokens but have the same type.
        E.g.`Box<dyn T>` and `Box<dyn S>` but both `S` and `T` have the `Clone` trait bound.
- [ ] Non-public structs. I.e. structs that aren't fully internally visible.
    This will lead to an compiler-error but isn't cought while running this macro.
    This might be infeasible?

#### Unsolvable problems

These are problems that are either impossible to solve or very infeasible.
For instance, something infeasible would be to parse all files for a full type resolution of a given crate.
That would be a job for the compiler in a later stage.

- Structs that are altered or generated by other macros.
- Type comparison and type resolution. E.g. `type test = Option<String>` won't be detected as optional.
    The current type checks are literal comparisons of the type tokens.
