use std::fmt::Debug;

pub enum Error {
    NoMergeCommit {
        r#for: String,
    },
    PicoArgsError(pico_args::Error),
    FindCommitError {
        sha: String,
        source: git2::Error,
    },
    GraphDescendantOfError {
        commit: String,
        ancestory: String,
        source: git2::Error,
    },
    RevParseError {
        sha: String,
        source: git2::Error,
    },
    ExpectedCommitError {
        sha: String,
    },
    FindRemoteError {
        remote: String,
        source: git2::Error,
    },
    Git2Error(git2::Error),
}

impl std::fmt::Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::NoMergeCommit { r#for } => {
                write!(f, "No merge commit for {} found", r#for)
            }
            Error::PicoArgsError(e) => write!(f, "{}", e),
            Error::FindCommitError { sha, source: _ } => {
                write!(f, "Failed to find commit {}", sha)
            }
            Error::GraphDescendantOfError {
                commit,
                ancestory,
                source: _,
            } => {
                write!(
                    f,
                    "Failed to determine of {} is a descendant of {}",
                    commit, ancestory,
                )
            }
            Error::RevParseError { sha, source: _ } => {
                write!(f, "Failed to parse revision {}", sha)
            }
            Error::ExpectedCommitError { sha } => {
                write!(f, "Expected {} to be a commit", sha)
            }
            Error::FindRemoteError { remote, source: _ } => {
                write!(f, "Failed to find remote {}", remote)
            }
            Error::Git2Error(e) => write!(f, "{}", e),
        }
    }
}

impl Debug for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        writeln!(f, "{}", self)?;
        writeln!(f)?;

        let mut maybe_e = std::error::Error::source(self);
        while let Some(e) = maybe_e {
            writeln!(f, "{}", e)?;
            maybe_e = std::error::Error::source(e);
        }
        Ok(())
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match self {
            Error::NoMergeCommit { r#for: _ } => None,
            Error::PicoArgsError(e) => Some(e),
            Error::FindCommitError { sha: _, source } => Some(source),
            Error::GraphDescendantOfError {
                commit: _,
                ancestory: _,
                source,
            } => Some(source),
            Error::RevParseError { sha: _, source } => Some(source),
            Error::ExpectedCommitError { sha: _ } => None,
            Error::FindRemoteError { remote: _, source } => Some(source),
            Error::Git2Error(e) => e.source(),
        }
    }

    fn cause(&self) -> Option<&dyn std::error::Error> {
        self.source()
    }
}

impl From<pico_args::Error> for Error {
    fn from(e: pico_args::Error) -> Self {
        Self::PicoArgsError(e)
    }
}

impl From<git2::Error> for Error {
    fn from(e: git2::Error) -> Self {
        Self::Git2Error(e)
    }
}

pub type Result<T> = std::result::Result<T, Error>;
