use genetic_algorithm::evolve::prelude::*;
use rand::prelude::*;
use rand::rngs::SmallRng;

// see https://en.wikipedia.org/wiki/Eight_queens_puzzle
#[derive(Clone, Debug)]
struct NQueensFitness;
impl Fitness for NQueensFitness {
    type Genotype = UniqueDiscreteGenotype<u8>;
    fn calculate_for_chromosome(
        &mut self,
        chromosome: &Chromosome<Self::Genotype>,
    ) -> Option<FitnessValue> {
        let mut score = 0;
        let max_index = chromosome.genes.len() - 1;
        for i in 0..max_index {
            for j in 0..max_index {
                if i != j {
                    let dx = i.abs_diff(j);
                    let dy = chromosome.genes[i].abs_diff(chromosome.genes[j]) as usize;
                    if dx == dy {
                        //diagonal clash
                        score += 1;
                    }
                }
            }
        }
        Some(score)
    }
}

fn main() {
    const BOARD_SIZE: u8 = 64;

    let mut rng = SmallRng::from_entropy();
    let genotype = UniqueDiscreteGenotype::builder()
        .with_allele_values((0..BOARD_SIZE).collect())
        .build()
        .unwrap();

    println!("{}", genotype);

    let evolve = Evolve::builder()
        .with_genotype(genotype)
        .with_population_size(20)
        .with_max_stale_generations(10000)
        .with_fitness_ordering(FitnessOrdering::Minimize)
        .with_target_fitness_score(0)
        .with_mutate(MutateOnce(0.2))
        .with_fitness(NQueensFitness)
        .with_crossover(CrossoverClone(true))
        .with_compete(CompeteElite)
        .call(&mut rng)
        .unwrap();

    println!("{}", evolve);

    if let Some(best_chromosome) = evolve.best_chromosome {
        for gene in best_chromosome.genes {
            let mut chars: Vec<char> = (0..BOARD_SIZE).map(|_| '.').collect();
            chars[gene as usize] = 'X';
            println!("{}", String::from_iter(chars));
        }
    }
}
