/// Writes specified number of bits of a given value to a slice at a specified
/// offset in portable way.
///
/// This macro is used by [`ArchiveBuilder`] to serialize data to storage. It
/// ensures that the data is written in a portable way independent of the
/// platform and compiler. To read the data written by this macro, use
/// [`read_bytes`].
///
/// The maximum amount of bits which can be written is 64. If `$num_bits` is
/// not specified, [`std::mem::size_of::<T>`] is used.
///
/// # Arguments
///
/// * `T` – integer type of the value to write (signed or unsigned),
/// * `value` – value to write,
/// * `data` – slice for writing to,
/// * `offset` – offset in bits in the slice, where the value should be written,
/// * `num_bits` – how many bits of the value to write.
///
/// [`ArchiveBuilder`]: trait.ArchiveBuilder.html
/// [`read_bytes`]: macro.read_bytes.html
/// [`std::mem::size_of::<T>`]: https://doc.rust-lang.org/std/mem/fn.size_of.html
#[macro_export]
macro_rules! write_bytes {
    ($T:tt; $value:expr, $data:expr, $offset:expr, $num_bits:expr) => {{
        debug_assert!($num_bits <= 64, "num_bits cannot be > 64");

        let destination = &mut $data[$offset / 8] as *mut u8;
        let bit_offset = $offset % 8;

        let mut value_to_store = $value as u64;
        if <$T as $crate::helper::Int>::IS_SIGNED {
            value_to_store = masked!(value_to_store, $num_bits);
        }
        let mut value_mask = masked!(u64::max_value(), $num_bits);
        let mut value: u64 = 0;

        // read previous data from destination to value
        unsafe {
            ::std::ptr::copy(
                destination,
                &mut value as *mut u64 as *mut u8,
                num_bytes!($offset, $num_bits),
            )
        };
        value = u64::from_le(value);

        // remove previous value, but keep surrounding data
        value &= !(value_mask << bit_offset);
        // add new value, and keep surrounding data
        value |= value_to_store << bit_offset;

        // write back the new data
        value = u64::to_le(value);
        unsafe {
            ::std::ptr::copy(
                &value as *const u64 as *const u8,
                destination,
                num_bytes!($offset, $num_bits),
            )
        };

        // one byte might be missing
        let batched_bits_written = num_bytes!($offset, $num_bits) * 8 - bit_offset;
        if batched_bits_written < $num_bits {
            let destination = &mut $data[num_bytes!($offset, $num_bits)] as *mut u8;
            value_to_store >>= batched_bits_written % (num_bytes!($offset, $num_bits) * 8);
            value_mask >>= batched_bits_written % (num_bytes!($offset, $num_bits) * 8);
            unsafe {
                // remove previous value, but keep surrounding data
                let byte_value = *destination & !value_mask as u8;
                // add new value, and keep surrounding data
                *destination = byte_value | value_to_store as u8;
            }
        }
    }};
    ($T:tt; $value:expr, $data:expr, $offset:expr) => {
        write_bytes!($T; $value, $data, $offset, ::std::mem::size_of::<$T>() * 8)
    };
    ($T:tt; $value:expr, $data:expr) => {
        write_bytes!($T, $value, $data, 0, ::std::mem::size_of::<$T>() * 8)
    };
}

#[cfg(test)]
mod test {
    fn test_writer(value: u64, offset: usize, num_bits: usize, expected: &[u8]) {
        let mut buffer = vec![0u8; expected.len()];
        write_bytes!(u64; value, &mut buffer[..], offset, num_bits);
        assert_eq!(buffer, expected);
    }

    #[test]
    #[rustfmt::skip]
    fn byte_writer_unsigned_test() {
        test_writer(1, 8, 8, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 7, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 6, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 5, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 4, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 3, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 2, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 1, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 8, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 7, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 6, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 5, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 4, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 3, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 2, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 1, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 8, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 7, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 6, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 5, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 4, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 3, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 2, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 1, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 8, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 7, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 6, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 5, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 4, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 3, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 2, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 1, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 8, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 7, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 6, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 5, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 4, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 3, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 2, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 1, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 8, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 7, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 6, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 5, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 4, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 3, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 2, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 1, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 8, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 7, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 6, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 5, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 4, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 3, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 2, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 1, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 8, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 7, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 6, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 5, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 4, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 3, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 2, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 1, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 8, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 7, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 6, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 5, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 4, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 3, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 2, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 1, &vec![1, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 16, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 15, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 14, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 13, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 12, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 11, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 10, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 9, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 8, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 7, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 6, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 5, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 4, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 3, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 2, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 1, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 16, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 15, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 14, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 13, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 12, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 11, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 10, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 9, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 8, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 7, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 6, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 5, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 4, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 3, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 2, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 1, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 16, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 15, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 14, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 13, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 12, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 11, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 10, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 9, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 8, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 7, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 6, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 5, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 4, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 3, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 2, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 1, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 16, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 15, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 14, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 13, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 12, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 11, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 10, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 9, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 8, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 7, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 6, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 5, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 4, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 3, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 2, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 1, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 16, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 15, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 14, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 13, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 12, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 11, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 10, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 9, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 8, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 7, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 6, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 5, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 4, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 3, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 2, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 1, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 16, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 15, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 14, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 13, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 12, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 11, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 10, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 9, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 8, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 7, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 6, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 5, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 4, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 3, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 2, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 1, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 16, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 15, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 14, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 13, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 12, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 11, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 10, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 9, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 8, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 7, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 6, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 5, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 4, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 3, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 2, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 1, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 16, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 15, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 14, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 13, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 12, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 11, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 10, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 9, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 8, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 7, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 6, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 5, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 4, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 3, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 2, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 1, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 16, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 15, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 14, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 13, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 12, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 11, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 10, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 9, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 8, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 7, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 6, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 5, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 4, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 3, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 2, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 1, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 32, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 31, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 30, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 29, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 28, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 27, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 26, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 25, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 24, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 23, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 22, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 21, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 20, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 19, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 18, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 17, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 16, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 15, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 14, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 13, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 12, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 11, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 10, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 9, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 8, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 7, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 6, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 5, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 4, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 3, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 2, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 1, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 32, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 31, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 30, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 29, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 28, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 27, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 26, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 25, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 24, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 23, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 22, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 21, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 20, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 19, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 18, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 17, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 16, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 15, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 14, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 13, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 12, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 11, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 10, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 9, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 8, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 7, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 6, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 5, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 4, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 3, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 2, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 1, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 32, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 31, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 30, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 29, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 28, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 27, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 26, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 25, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 24, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 23, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 22, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 21, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 20, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 19, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 18, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 17, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 16, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 15, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 14, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 13, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 12, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 11, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 10, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 9, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 8, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 7, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 6, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 5, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 4, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 3, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 2, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 1, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 32, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 31, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 30, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 29, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 28, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 27, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 26, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 25, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 24, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 23, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 22, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 21, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 20, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 19, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 18, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 17, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 16, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 15, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 14, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 13, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 12, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 11, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 10, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 9, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 8, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 7, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 6, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 5, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 4, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 3, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 2, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 1, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 32, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 31, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 30, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 29, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 28, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 27, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 26, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 25, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 24, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 23, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 22, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 21, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 20, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 19, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 18, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 17, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 16, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 15, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 14, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 13, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 12, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 11, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 10, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 9, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 8, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 7, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 6, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 5, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 4, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 3, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 2, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 1, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 32, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 31, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 30, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 29, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 28, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 27, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 26, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 25, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 24, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 23, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 22, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 21, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 20, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 19, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 18, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 17, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 16, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 15, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 14, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 13, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 12, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 11, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 10, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 9, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 8, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 7, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 6, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 5, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 4, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 3, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 2, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 1, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 32, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 31, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 30, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 29, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 28, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 27, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 26, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 25, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 24, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 23, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 22, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 21, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 20, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 19, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 18, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 17, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 16, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 15, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 14, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 13, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 12, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 11, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 10, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 9, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 8, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 7, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 6, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 5, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 4, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 3, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 2, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 1, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 32, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 31, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 30, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 29, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 28, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 27, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 26, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 25, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 24, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 23, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 22, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 21, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 20, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 19, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 18, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 17, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 16, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 15, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 14, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 13, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 12, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 11, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 10, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 9, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 8, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 7, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 6, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 5, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 4, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 3, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 2, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 1, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 32, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 31, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 30, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 29, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 28, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 27, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 26, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 25, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 24, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 23, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 22, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 21, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 20, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 19, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 18, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 17, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 16, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 15, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 14, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 13, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 12, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 11, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 10, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 9, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 8, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 7, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 6, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 5, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 4, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 3, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 2, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 1, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 64, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 63, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 62, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 61, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 60, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 59, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 58, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 57, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 56, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 55, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 54, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 53, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 52, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 51, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 50, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 49, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 48, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 47, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 46, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 45, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 44, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 43, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 42, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 41, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 40, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 39, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 38, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 37, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 36, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 35, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 34, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 33, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 32, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 31, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 30, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 29, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 28, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 27, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 26, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 25, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 24, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 23, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 22, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 21, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 20, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 19, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 18, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 17, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 16, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 15, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 14, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 13, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 12, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 11, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 10, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 9, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 8, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 7, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 6, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 5, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 4, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 3, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 2, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 8, 1, &vec![0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 64, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 63, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 62, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 61, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 60, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 59, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 58, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 57, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 56, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 55, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 54, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 53, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 52, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 51, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 50, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 49, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 48, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 47, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 46, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 45, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 44, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 43, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 42, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 41, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 40, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 39, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 38, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 37, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 36, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 35, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 34, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 33, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 32, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 31, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 30, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 29, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 28, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 27, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 26, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 25, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 24, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 23, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 22, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 21, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 20, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 19, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 18, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 17, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 16, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 15, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 14, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 13, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 12, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 11, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 10, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 9, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 8, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 7, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 6, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 5, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 4, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 3, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 2, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 7, 1, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 64, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 63, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 62, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 61, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 60, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 59, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 58, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 57, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 56, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 55, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 54, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 53, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 52, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 51, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 50, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 49, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 48, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 47, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 46, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 45, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 44, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 43, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 42, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 41, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 40, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 39, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 38, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 37, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 36, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 35, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 34, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 33, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 32, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 31, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 30, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 29, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 28, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 27, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 26, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 25, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 24, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 23, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 22, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 21, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 20, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 19, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 18, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 17, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 16, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 15, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 14, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 13, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 12, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 11, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 10, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 9, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 8, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 7, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 6, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 5, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 4, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 3, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 2, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 6, 1, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 64, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 63, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 62, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 61, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 60, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 59, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 58, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 57, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 56, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 55, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 54, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 53, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 52, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 51, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 50, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 49, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 48, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 47, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 46, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 45, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 44, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 43, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 42, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 41, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 40, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 39, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 38, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 37, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 36, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 35, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 34, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 33, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 32, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 31, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 30, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 29, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 28, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 27, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 26, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 25, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 24, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 23, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 22, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 21, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 20, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 19, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 18, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 17, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 16, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 15, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 14, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 13, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 12, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 11, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 10, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 9, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 8, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 7, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 6, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 5, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 4, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 3, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 2, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 5, 1, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 64, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 63, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 62, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 61, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 60, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 59, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 58, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 57, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 56, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 55, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 54, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 53, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 52, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 51, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 50, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 49, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 48, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 47, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 46, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 45, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 44, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 43, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 42, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 41, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 40, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 39, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 38, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 37, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 36, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 35, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 34, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 33, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 32, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 31, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 30, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 29, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 28, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 27, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 26, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 25, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 24, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 23, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 22, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 21, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 20, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 19, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 18, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 17, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 16, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 15, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 14, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 13, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 12, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 11, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 10, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 9, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 8, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 7, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 6, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 5, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 4, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 3, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 2, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 4, 1, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 64, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 63, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 62, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 61, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 60, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 59, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 58, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 57, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 56, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 55, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 54, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 53, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 52, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 51, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 50, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 49, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 48, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 47, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 46, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 45, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 44, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 43, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 42, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 41, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 40, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 39, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 38, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 37, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 36, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 35, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 34, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 33, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 32, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 31, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 30, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 29, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 28, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 27, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 26, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 25, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 24, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 23, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 22, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 21, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 20, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 19, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 18, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 17, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 16, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 15, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 14, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 13, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 12, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 11, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 10, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 9, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 8, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 7, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 6, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 5, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 4, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 3, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 2, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 3, 1, &vec![8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 64, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 63, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 62, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 61, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 60, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 59, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 58, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 57, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 56, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 55, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 54, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 53, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 52, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 51, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 50, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 49, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 48, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 47, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 46, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 45, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 44, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 43, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 42, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 41, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 40, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 39, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 38, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 37, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 36, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 35, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 34, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 33, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 32, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 31, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 30, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 29, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 28, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 27, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 26, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 25, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 24, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 23, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 22, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 21, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 20, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 19, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 18, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 17, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 16, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 15, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 14, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 13, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 12, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 11, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 10, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 9, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 8, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 7, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 6, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 5, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 4, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 3, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 2, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 2, 1, &vec![4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 64, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 63, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 62, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 61, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 60, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 59, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 58, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 57, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 56, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 55, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 54, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 53, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 52, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 51, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 50, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 49, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 48, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 47, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 46, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 45, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 44, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 43, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 42, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 41, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 40, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 39, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 38, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 37, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 36, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 35, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 34, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 33, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 32, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 31, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 30, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 29, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 28, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 27, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 26, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 25, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 24, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 23, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 22, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 21, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 20, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 19, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 18, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 17, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 16, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 15, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 14, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 13, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 12, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 11, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 10, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 9, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 8, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 7, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 6, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 5, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 4, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 3, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 2, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 1, 1, &vec![2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 64, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 63, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 62, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 61, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 60, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 59, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 58, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 57, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 56, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 55, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 54, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 53, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 52, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 51, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 50, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 49, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 48, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 47, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 46, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 45, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 44, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 43, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 42, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 41, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 40, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 39, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 38, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 37, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 36, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 35, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 34, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 33, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 32, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 31, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 30, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 29, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 28, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 27, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 26, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 25, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 24, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 23, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 22, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 21, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 20, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 19, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 18, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 17, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 16, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 15, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 14, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 13, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 12, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 11, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 10, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 9, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 8, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 7, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 6, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 5, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 4, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 3, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 2, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(1, 0, 1, &vec![1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_writer(u64::max_value(), 7, 64, &vec![128, 255, 255, 255, 255, 255, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0]);
    }

    fn test_reader_unsigned(value: i64, offset: usize, num_bits: usize, expected: &[u8]) {
        let mut buffer = vec![0u8; expected.len()];
        write_bytes!(i64; value, &mut buffer[..], offset, num_bits);
        assert_eq!(buffer, expected);
    }

    #[test]
    #[rustfmt::skip]
    fn byte_writer_signed_test() {
        test_reader_unsigned(-1, 3, 16, &vec![0xf8, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-2, 3, 16, &vec![0xf0, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-4, 3, 16, &vec![0xe0, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-8, 3, 16, &vec![0xc0, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-16, 3, 16, &vec![0x80, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-32, 3, 16, &vec![0, 0xff, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-64, 3, 16, &vec![0, 0xfe, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-128, 3, 16, &vec![0, 0xfc, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-256, 3, 16, &vec![0, 0xf8, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-512, 3, 16, &vec![0, 0xf0, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-1024, 3, 16, &vec![0, 0xe0, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-2048, 3, 16, &vec![0, 0xc0, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-4096, 3, 16, &vec![0, 0x80, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-8192, 3, 16, &vec![0, 0, 0x07, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(-16384, 3, 16, &vec![0, 0, 0x06, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(1, 3, 16, &vec![0x08, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(2, 3, 16, &vec![0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(4, 3, 16, &vec![0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(8, 3, 16, &vec![0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(16, 3, 16, &vec![0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(32, 3, 16, &vec![0, 0x01, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(64, 3, 16, &vec![0, 0x02, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(128, 3, 16, &vec![0, 0x04, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(256, 3, 16, &vec![0, 0x08, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(512, 3, 16, &vec![0, 0x10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(1024, 3, 16, &vec![0, 0x20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(2048, 3, 16, &vec![0, 0x40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(4096, 3, 16, &vec![0, 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(8192, 3, 16, &vec![0, 0, 0x01, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(16384, 3, 16, &vec![0, 0, 0x02, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
        test_reader_unsigned(0, 3, 2, &vec![0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]);
    }
}
