/**
    Errors Rust (Error Mapping)
    Copyright (C) 2019  Jonathan Franco, Hebert Vera

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
**/

use std::error::Error;

use serde::{Serialize, Deserialize};

#[derive(Debug, Serialize, Deserialize)]
pub struct RustError {
    pub code: i64,
    pub message: String,
    pub description: String
}

pub struct RustErrorParams {
    pub code: Option<i64>,
    pub message: Option<String>,
    pub description: Option<String>
}

impl RustError {
    pub fn new(code: Option<i64>, message: Option<String>, description: Option<String>) -> RustError {
        let code_value: i64;
        let message_value: String;
        let description_value: String;

        if code.is_some() {
            code_value = code.unwrap();
        } else {
            code_value = 0;
        }

        if message.is_some() {
            message_value = message.unwrap();
        } else {
            message_value = String::new();
        }

        if description.is_some() {
            description_value = description.unwrap();
        } else {
            description_value = String::new();
        }

        RustError {
            code: code_value,
            message: message_value,
            description: description_value
        }
    }
    pub fn from_params(params: RustErrorParams) -> RustError {
        if params.code.is_some() || params.message.is_some() || params.description.is_some() {
            RustError::new(params.code, params.message, params.description)
        } else {
            panic!("You need to use params")
        }
    }

    pub fn from_string(description: &str) -> RustError {
        RustError::new(None, None, Some(String::from(description)))
    }

}

impl std::fmt::Display for RustError {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> Result<(), std::fmt::Error> {
        write!(f, "{:?}", self)
    }
}

impl Error for RustError {
    fn description(&self) -> &str {
        &self.description
    }
}