use crate::model::raw_file::RawFile;

macro_rules! assert_hashed_files {
    ($snapshot_name:expr, $raw_files:expr) => {
        assert_hashed_files_with_ignore_line_regex!($snapshot_name, vec![], $raw_files);
    };
}

macro_rules! assert_hashed_files_with_ignore_line_regex {
    ($snapshot_name:expr, $ignore_line_regex:expr, $raw_files:expr) => {
        let indexed_files =
            crate::model::indexed_file::to_indexed_files($ignore_line_regex, $raw_files);
        let hashed_files = crate::model::hashed_file::to_hashed_files(&indexed_files);

        // Then
        insta::assert_json_snapshot!($snapshot_name, hashed_files);
    };
}

#[test]
fn test_empty_indexed_files() {
    // Given
    let raw_files = vec![];

    // When/Then
    assert_hashed_files!("empty_indexed_files", raw_files);
}

#[test]
fn test_indexed_file_with_no_lines() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![],
    }];

    // When/Then
    assert_hashed_files!("indexed_file_with_no_lines", raw_files);
}

#[test]
fn test_indexed_file_hashing_computed_correctly() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![
            "line 1".to_string(),
            "line 2".to_string(),
            "line 3".to_string(),
        ],
    }];

    // When/Then
    assert_hashed_files!("indexed_file_hashing_computed_correctly", raw_files);
}

#[test]
fn test_indexed_file_with_ignore_line_regex() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![
            "line 1".to_string(),
            "line 2".to_string(),
            "line 3".to_string(),
        ],
    }];
    let ignore_line_regex = vec!["^line 1".to_string()];

    // When/Then
    assert_hashed_files_with_ignore_line_regex!(
        "indexed_file_with_ignore_line_regex",
        ignore_line_regex,
        raw_files
    );
}

#[test]
fn test_indexed_file_with_duplicate_lines() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec!["line".to_string(), "line".to_string(), "line ".to_string()],
    }];

    // When/Then
    assert_hashed_files!("indexed_file_with_duplicate_lines", raw_files);
}
