use std::collections::HashMap;

use crate::model::indexed_file::*;

#[cfg(test)]
use serde::{Serialize, Serializer};

use super::*;

pub type HashedFiles = Vec<HashedFile>;
pub type HashToLineNumber = HashMap<blake3::Hash, LineNumber>;
pub type LineNumberToHash = HashMap<LineNumber, blake3::Hash>;

#[cfg_attr(test, derive(Serialize))]
pub struct HashedFile {
    pub filename: Filename,
    pub number_of_lines: LineNumber,
    #[cfg_attr(test, serde(serialize_with = "ordered_hash_to_line_number"))]
    pub hash_to_line_number: HashToLineNumber,
    #[cfg_attr(test, serde(serialize_with = "ordered_line_number_to_hash"))]
    pub line_number_to_hash: LineNumberToHash,
}

#[cfg(test)]
fn ordered_hash_to_line_number<S>(
    value: &HashToLineNumber,
    serializer: S,
) -> Result<S::Ok, S::Error>
where
    S: Serializer,
{
    let ordered: std::collections::BTreeMap<String, &usize> = value
        .iter()
        .map(|(k, v)| (k.to_hex().to_string(), v))
        .collect();
    ordered.serialize(serializer)
}

#[cfg(test)]
fn ordered_line_number_to_hash<S>(
    value: &LineNumberToHash,
    serializer: S,
) -> Result<S::Ok, S::Error>
where
    S: Serializer,
{
    let ordered: std::collections::BTreeMap<&usize, String> = value
        .iter()
        .map(|(k, v)| (k, v.to_hex().to_string()))
        .collect();
    ordered.serialize(serializer)
}

impl HashedFile {
    pub fn new(filename: &str, indexed_file: &IndexedFile) -> Self {
        let mut hash_to_line_number = HashMap::new();
        let mut line_number_to_hash = HashMap::new();

        for (line_number, line) in indexed_file.line_number_to_line.clone() {
            let hash = crate::hashing::get_blake3_hash(&line);
            hash_to_line_number.insert(hash, line_number);
            line_number_to_hash.insert(line_number, hash);
        }

        HashedFile {
            filename: filename.to_string(),
            number_of_lines: indexed_file.number_of_lines,
            hash_to_line_number,
            line_number_to_hash,
        }
    }
}

pub fn to_hashed_files(indexed_files: &IndexedFiles) -> HashedFiles {
    let mut hashed_files = vec![];

    for (filename, indexed_file) in indexed_files {
        hashed_files.push(HashedFile::new(filename, indexed_file));
    }

    hashed_files
}

#[cfg(test)]
mod tests;
