use crate::model::raw_file::RawFile;

macro_rules! assert_indexed_files {
    ($snapshot_name:expr, $raw_files:expr) => {
        assert_indexed_files_base!($snapshot_name, vec![], vec![], $raw_files);
    };
}

macro_rules! assert_indexed_files_with_ignore_line_regex {
    ($snapshot_name:expr, $ignore_line_regex:expr, $raw_files:expr) => {
        assert_indexed_files_base!($snapshot_name, vec![], $ignore_line_regex, $raw_files);
    };
}

macro_rules! assert_indexed_files_with_ignore_file_regex {
    ($snapshot_name:expr, $ignore_file_regex:expr, $raw_files:expr) => {
        assert_indexed_files_base!($snapshot_name, $ignore_file_regex, vec![], $raw_files);
    };
}

macro_rules! assert_indexed_files_base {
    ($snapshot_name:expr, $ignore_file_regex:expr, $ignore_line_regex:expr, $raw_files:expr) => {
        let indexed_files = crate::model::indexed_file::to_indexed_files(
            $ignore_file_regex,
            $ignore_line_regex,
            $raw_files,
        );

        // Then
        insta::assert_json_snapshot!($snapshot_name, indexed_files);
    };
}

#[test]
fn test_empty_raw_files() {
    // Given
    let raw_files = vec![];

    // When/Then
    assert_indexed_files!("empty_raw_files", raw_files);
}

#[test]
fn test_raw_file_with_no_lines() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![],
    }];

    // When/Then
    assert_indexed_files!("raw_file_with_no_lines", raw_files);
}

#[test]
fn test_indexed_file_stores_correct_line_number() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![
            "line 1".to_string(),
            "line 2".to_string(),
            "line 3".to_string(),
        ],
    }];

    // When/Then
    assert_indexed_files!("indexed_file_stores_correct_line_number", raw_files);
}

#[test]
fn test_indexed_files_consistent_ordering() {
    // Given
    let raw_files_1 = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 4".to_string(),
                "line 5".to_string(),
                "line 6".to_string(),
            ],
        },
    ];
    let raw_files_2 = vec![
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 4".to_string(),
                "line 5".to_string(),
                "line 6".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
            ],
        },
    ];

    // When/Then
    assert_indexed_files!("indexed_files_consistent_ordering", raw_files_1);
    assert_indexed_files!("indexed_files_consistent_ordering", raw_files_2);
}

#[test]
fn test_indexed_file_ignore_line_regex() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![
            "line 1".to_string(),
            "line 2".to_string(),
            "line 3".to_string(),
        ],
    }];
    let ignore_line_regex = vec!["^line 1".to_string()];

    // When/Then
    assert_indexed_files_with_ignore_line_regex!(
        "indexed_file_ignore_line_regex",
        ignore_line_regex,
        raw_files
    );
}

#[test]
fn test_indexed_file_ignore_file_regex() {
    // Given
    let raw_files = vec![RawFile {
        filename: "ignore_file.stx".to_string(),
        lines: vec![
            "line 1".to_string(),
            "line 2".to_string(),
            "line 3".to_string(),
        ],
    }];
    let ignore_file_regex = vec!["[.]stx$".to_string()];

    // When/Then
    assert_indexed_files_with_ignore_file_regex!(
        "indexed_file_ignore_file_regex",
        ignore_file_regex,
        raw_files
    );
}
