use std::collections::{BTreeMap, HashMap};

#[cfg(test)]
use serde::{Serialize, Serializer};

use crate::model::raw_file::*;

use super::*;

pub type LineNumberToLine = HashMap<LineNumber, Line>;
pub type IndexedFiles = BTreeMap<Filename, IndexedFile>;

#[cfg_attr(test, derive(Serialize))]
pub struct IndexedFile {
    pub number_of_lines: LineNumber,
    #[cfg_attr(test, serde(serialize_with = "ordered_line_number_to_line"))]
    pub line_number_to_line: LineNumberToLine,
}

#[cfg(test)]
fn ordered_line_number_to_line<S>(
    value: &LineNumberToLine,
    serializer: S,
) -> Result<S::Ok, S::Error>
where
    S: Serializer,
{
    let ordered: std::collections::BTreeMap<_, _> = value.iter().collect();
    ordered.serialize(serializer)
}

impl IndexedFile {
    pub fn new(raw_file: RawFile) -> Self {
        let mut line_number_to_line = HashMap::new();
        let number_of_lines = raw_file.lines.len();
        let mut line_number = 1;

        for line in raw_file.lines {
            let trimmed_line = line.trim().to_string();

            line_number_to_line.insert(line_number, trimmed_line);

            line_number += 1;
        }

        IndexedFile {
            number_of_lines,
            line_number_to_line,
        }
    }
}

pub fn to_indexed_files(raw_files: RawFiles) -> IndexedFiles {
    let mut indexed_files = BTreeMap::new();

    for raw_file in raw_files {
        let filename = raw_file.filename.clone();
        indexed_files.insert(filename, IndexedFile::new(raw_file));
    }

    indexed_files
}

#[cfg(test)]
mod tests;
