use crate::model::raw_file::RawFile;

macro_rules! assert_duplicates {
    ($snapshot_name:expr, $raw_files:expr) => {
        let indexed_files = crate::model::indexed_file::to_indexed_files($raw_files);
        let hashed_files = crate::model::hashed_file::to_hashed_files(&indexed_files);
        let duplicates = crate::duplicate_detection::get_all_duplicates(hashed_files);

        // Then
        insta::assert_json_snapshot!($snapshot_name, duplicates);
    };
}

#[test]
fn test_empty_hashed_files() {
    // Given
    let raw_files = vec![];

    // When/Then
    assert_duplicates!("empty_hashed_files", raw_files);
}

#[test]
fn test_hashed_file_with_no_lines() {
    // Given
    let raw_files = vec![RawFile {
        filename: "raw_file_filename".to_string(),
        lines: vec![],
    }];

    // When/Then
    assert_duplicates!("hashed_file_with_no_lines", raw_files);
}

#[test]
fn test_hashed_files_with_no_lines() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![],
        },
    ];

    // When/Then
    assert_duplicates!("hashed_files_with_no_lines", raw_files);
}

#[test]
fn test_hashed_files_with_no_duplicate() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 4".to_string(),
                "line 5".to_string(),
                "line 6".to_string(),
            ],
        },
    ];

    // When/Then
    assert_duplicates!("hashed_files_with_no_duplicate", raw_files);
}

#[test]
fn test_hashed_files_with_duplicate() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
            ],
        },
    ];

    // When/Then
    assert_duplicates!("hashed_files_with_duplicate", raw_files);
}

#[test]
fn test_hashed_files_with_duplicate_on_different_lines() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "precursor 1".to_string(),
                "precursor 2".to_string(),
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "line 4".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "successor 1".to_string(),
            ],
        },
    ];

    // When/Then
    assert_duplicates!("hashed_files_with_duplicate_on_different_lines", raw_files);
}

#[test]
fn test_hashed_files_and_empty_with_duplicate_on_different_lines() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "precursor 1".to_string(),
                "precursor 2".to_string(),
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "line 4".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![],
        },
        RawFile {
            filename: "raw_file_filename_3".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "successor 1".to_string(),
            ],
        },
    ];

    // When/Then
    assert_duplicates!(
        "hashed_files_and_empty_with_duplicate_on_different_lines",
        raw_files
    );
}

#[test]
fn test_hashed_files_with_multiple_duplicate_in_different_files() {
    // Given
    let raw_files = vec![
        RawFile {
            filename: "raw_file_filename_1".to_string(),
            lines: vec![
                "precursor 1".to_string(),
                "precursor 2".to_string(),
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "line 4".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_2".to_string(),
            lines: vec![
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "successor 1".to_string(),
            ],
        },
        RawFile {
            filename: "raw_file_filename_3".to_string(),
            lines: vec![
                "precursor 1".to_string(),
                "line 1".to_string(),
                "line 2".to_string(),
                "line 3".to_string(),
                "successor 1".to_string(),
            ],
        },
    ];

    // When/Then
    assert_duplicates!(
        "hashed_files_with_multiple_duplicate_in_different_files",
        raw_files
    );
}
