use std::collections::HashMap;

pub type LineNumber = usize;
pub type Filename = String;
pub type Line = String;
pub type Lines = Vec<Line>;
pub type RawFiles = Vec<RawFile>;
pub type HashedFiles = Vec<HashedFile>;
pub type Duplicates = Vec<Duplicate>;
pub type HashToLineNumber = HashMap<blake3::Hash, LineNumber>;
pub type LineNumberToHash = HashMap<LineNumber, blake3::Hash>;
pub type LineNumberToLine = HashMap<LineNumber, Line>;
pub type FilenameToLineNumberToLine = HashMap<Filename, LineNumberToLine>;

pub struct HashedFile {
    pub filename: Filename,
    pub number_of_lines: usize,
    pub hash_to_line_number: HashToLineNumber,
    pub line_number_to_hash: LineNumberToHash,
}

impl HashedFile {
    pub fn new(raw_file: RawFile) -> (HashedFile, LineNumberToLine) {
        let mut hash_to_line_number = HashMap::new();
        let mut line_number_to_hash = HashMap::new();
        let mut line_number_to_line = HashMap::new();
        let number_of_lines = raw_file.lines.len();
        let mut line_number = 1;

        for line in raw_file.lines {
            let trimmed_line = line.trim().to_string();
            let hash = crate::hashing::get_blake3_hash(&trimmed_line);

            hash_to_line_number.insert(hash, line_number);
            line_number_to_hash.insert(line_number, hash);
            line_number_to_line.insert(line_number, trimmed_line);

            line_number += 1;
        }

        (
            HashedFile {
                filename: raw_file.filename,
                number_of_lines,
                hash_to_line_number,
                line_number_to_hash,
            },
            line_number_to_line,
        )
    }
}

pub fn to_file_content(raw_files: RawFiles) -> (HashedFiles, FilenameToLineNumberToLine) {
    let mut hashed_files = vec![];
    let mut filename_to_line_number_to_line = HashMap::new();

    for raw_file in raw_files {
        let filename = raw_file.filename.clone();
        let (hashed_file, line_number_to_line) = HashedFile::new(raw_file);
        hashed_files.push(hashed_file);
        filename_to_line_number_to_line.insert(filename, line_number_to_line);
    }

    (hashed_files, filename_to_line_number_to_line)
}

pub struct RawFile {
    pub filename: Filename,
    pub lines: Lines,
}

#[derive(Debug)]
pub struct LineRange {
    pub start: LineNumber,
    pub end: LineNumber,
}

#[derive(Debug)]
pub struct Duplicate {
    pub file_1: Filename,
    pub file_1_line_range: LineRange,
    pub file_2: Filename,
    pub file_2_line_range: LineRange,
}
