use base64::encode;
use image::*;
use lcs_diff::DiffResult;

fn encode_image(image: &DynamicImage) -> Vec<String> {
    let pixels = image.raw_pixels();
    pixels
        .chunks(image.width() as usize * 3)
        .map(|chunk| encode(chunk))
        .collect()
}

/// Compute the amount of difference between two images
///
/// Receives two `&DymanicImage` (the images that are going to be compared)
/// and returns a `f64` between 0 and 1, representing how much both images
/// differ from each other, i.e. 0 means they are completely equal and 1 means
/// they are completely different.
///
/// # Example
/// ```no_compile
/// use image::*;
/// use diffimg;
///
/// fn main() {
///     let image1 = image::open("tests/images/image1.png").expect("Failed to open image");
///     let image2 = image::open("tests/images/image2.png").expect("Failed to open image");
///     
///     let diff = diffimg::diff(&image1, &image2);
/// }
/// ```
pub fn diff(image_a: &DynamicImage, image_b: &DynamicImage) -> f64 {
    let encoded_image_a = encode_image(image_a);
    let encoded_image_b = encode_image(image_b);
    let diff_result = lcs_diff::diff(&encoded_image_a, &encoded_image_b);

    let total = diff_result.len() as f64;
    let mut changes = 0;
    for result in diff_result {
        match result {
            DiffResult::Added(_) | DiffResult::Removed(_) => changes += 1,
            DiffResult::Common(_) => continue,
        }
    }

    f64::from(changes) / total
}

#[cfg(test)]
mod tests {

    #[test]
    fn diff_images() {
        let image1 = image::open("tests/images/image1.png").expect("Failed to open image!");
        let image2 = image::open("tests/images/image2.png").expect("Failed to open image!");

        let diff = super::diff(&image1, &image2);
        assert!(diff > 0.9)
    }
}
