use std::collections::{HashMap, HashSet};

use crate::config::Profile;
use anyhow::{Context, Result};
use sqlx::{
    mysql::{MySqlConnectOptions, MySqlSslMode},
    MySqlPool,
};

pub const KEYWORDS: [&'static str; 389] = [
    "ABS",
    "ACTION",
    "ADD",
    "ALL",
    "ALLOCATE",
    "ALTER",
    "AND",
    "ANY",
    "APPLY",
    "ARE",
    "ARRAY",
    "ARRAY_AGG",
    "ARRAY_MAX_CARDINALITY",
    "AS",
    "ASC",
    "ASENSITIVE",
    "ASSERT",
    "ASYMMETRIC",
    "AT",
    "ATOMIC",
    "AUTHORIZATION",
    "AVG",
    "AVRO",
    "BEGIN",
    "BEGIN_FRAME",
    "BEGIN_PARTITION",
    "BETWEEN",
    "BIGINT",
    "BINARY",
    "BLOB",
    "BOOLEAN",
    "BOTH",
    "BY",
    "BYTEA",
    "CALL",
    "CALLED",
    "CARDINALITY",
    "CASCADE",
    "CASCADED",
    "CASE",
    "CAST",
    "CEIL",
    "CEILING",
    "CHAIN",
    "CHAR",
    "CHARACTER",
    "CHARACTER_LENGTH",
    "CHAR_LENGTH",
    "CHECK",
    "CLOB",
    "CLOSE",
    "COALESCE",
    "COLLATE",
    "COLLECT",
    "COLUMN",
    "COLUMNS",
    "COMMIT",
    "COMMITTED",
    "CONDITION",
    "CONNECT",
    "CONSTRAINT",
    "CONTAINS",
    "CONVERT",
    "COPY",
    "CORR",
    "CORRESPONDING",
    "COUNT",
    "COVAR_POP",
    "COVAR_SAMP",
    "CREATE",
    "CROSS",
    "CSV",
    "CUBE",
    "CUME_DIST",
    "CURRENT",
    "CURRENT_CATALOG",
    "CURRENT_DATE",
    "CURRENT_DEFAULT_TRANSFORM_GROUP",
    "CURRENT_PATH",
    "CURRENT_ROLE",
    "CURRENT_ROW",
    "CURRENT_SCHEMA",
    "CURRENT_TIME",
    "CURRENT_TIMESTAMP",
    "CURRENT_TRANSFORM_GROUP_FOR_TYPE",
    "CURRENT_USER",
    "CURSOR",
    "CYCLE",
    "DATE",
    "DAY",
    "DEALLOCATE",
    "DEC",
    "DECIMAL",
    "DECLARE",
    "DEFAULT",
    "DELETE",
    "DENSE_RANK",
    "DEREF",
    "DESC",
    "DESCRIBE",
    "DETERMINISTIC",
    "DISCONNECT",
    "DISTINCT",
    "DOUBLE",
    "DROP",
    "DYNAMIC",
    "EACH",
    "ELEMENT",
    "ELSE",
    "END",
    "END_EXEC",
    "END_FRAME",
    "END_PARTITION",
    "EQUALS",
    "ERROR",
    "ESCAPE",
    "EVERY",
    "EXCEPT",
    "EXEC",
    "EXECUTE",
    "EXISTS",
    "EXP",
    "EXTENDED",
    "EXTERNAL",
    "EXTRACT",
    "FALSE",
    "FETCH",
    "FIELDS",
    "FILTER",
    "FIRST",
    "FIRST_VALUE",
    "FLOAT",
    "FLOOR",
    "FOLLOWING",
    "FOR",
    "FOREIGN",
    "FRAME_ROW",
    "FREE",
    "FROM",
    "FULL",
    "FUNCTION",
    "FUSION",
    "GET",
    "GLOBAL",
    "GRANT",
    "GROUP",
    "GROUPING",
    "GROUPS",
    "HAVING",
    "HEADER",
    "HOLD",
    "HOUR",
    "IDENTITY",
    "IF",
    "IN",
    "INDEX",
    "INDICATOR",
    "INNER",
    "INOUT",
    "INSENSITIVE",
    "INSERT",
    "INT",
    "INTEGER",
    "INTERSECT",
    "INTERSECTION",
    "INTERVAL",
    "INTO",
    "IS",
    "ISOLATION",
    "JOIN",
    "JSONFILE",
    "KEY",
    "LAG",
    "LANGUAGE",
    "LARGE",
    "LAST",
    "LAST_VALUE",
    "LATERAL",
    "LEAD",
    "LEADING",
    "LEFT",
    "LEVEL",
    "LIKE",
    "LIKE_REGEX",
    "LIMIT",
    "LISTAGG",
    "LN",
    "LOCAL",
    "LOCALTIME",
    "LOCALTIMESTAMP",
    "LOCATION",
    "LOWER",
    "MATCH",
    "MATERIALIZED",
    "MAX",
    "MEMBER",
    "MERGE",
    "METHOD",
    "MIN",
    "MINUTE",
    "MOD",
    "MODIFIES",
    "MODULE",
    "MONTH",
    "MULTISET",
    "NATIONAL",
    "NATURAL",
    "NCHAR",
    "NCLOB",
    "NEW",
    "NEXT",
    "NO",
    "NONE",
    "NORMALIZE",
    "NOT",
    "NTH_VALUE",
    "NTILE",
    "NULL",
    "NULLIF",
    "NULLS",
    "NUMERIC",
    "OBJECT",
    "OCCURRENCES_REGEX",
    "OCTET_LENGTH",
    "OF",
    "OFFSET",
    "OLD",
    "ON",
    "ONLY",
    "OPEN",
    "OR",
    "ORC",
    "ORDER",
    "OUT",
    "OUTER",
    "OVER",
    "OVERFLOW",
    "OVERLAPS",
    "OVERLAY",
    "PARAMETER",
    "PARQUET",
    "PARTITION",
    "PERCENT",
    "PERCENTILE_CONT",
    "PERCENTILE_DISC",
    "PERCENT_RANK",
    "PERIOD",
    "PORTION",
    "POSITION",
    "POSITION_REGEX",
    "POWER",
    "PRECEDES",
    "PRECEDING",
    "PRECISION",
    "PREPARE",
    "PRIMARY",
    "PROCEDURE",
    "RANGE",
    "RANK",
    "RCFILE",
    "READ",
    "READS",
    "REAL",
    "RECURSIVE",
    "REF",
    "REFERENCES",
    "REFERENCING",
    "REGCLASS",
    "REGR_AVGX",
    "REGR_AVGY",
    "REGR_COUNT",
    "REGR_INTERCEPT",
    "REGR_R2",
    "REGR_SLOPE",
    "REGR_SXX",
    "REGR_SXY",
    "REGR_SYY",
    "RELEASE",
    "RENAME",
    "REPEATABLE",
    "RESTRICT",
    "RESULT",
    "RETURN",
    "RETURNS",
    "REVOKE",
    "RIGHT",
    "ROLLBACK",
    "ROLLUP",
    "ROW",
    "ROWID",
    "ROWS",
    "ROW_NUMBER",
    "SAVEPOINT",
    "SCHEMA",
    "SCOPE",
    "SCROLL",
    "SEARCH",
    "SECOND",
    "SELECT",
    "SENSITIVE",
    "SEQUENCEFILE",
    "SERIALIZABLE",
    "SESSION",
    "SESSION_USER",
    "SET",
    "SHOW",
    "SIMILAR",
    "SMALLINT",
    "SOME",
    "SPECIFIC",
    "SPECIFICTYPE",
    "SQL",
    "SQLEXCEPTION",
    "SQLSTATE",
    "SQLWARNING",
    "SQRT",
    "START",
    "STATIC",
    "STDDEV_POP",
    "STDDEV_SAMP",
    "STDIN",
    "STORED",
    "SUBMULTISET",
    "SUBSTRING",
    "SUBSTRING_REGEX",
    "SUCCEEDS",
    "SUM",
    "SYMMETRIC",
    "SYSTEM",
    "SYSTEM_TIME",
    "SYSTEM_USER",
    "TABLE",
    "TABLESAMPLE",
    "TEXT",
    "TEXTFILE",
    "THEN",
    "TIES",
    "TIME",
    "TIMESTAMP",
    "TIMEZONE_HOUR",
    "TIMEZONE_MINUTE",
    "TO",
    "TOP",
    "TRAILING",
    "TRANSACTION",
    "TRANSLATE",
    "TRANSLATE_REGEX",
    "TRANSLATION",
    "TREAT",
    "TRIGGER",
    "TRIM",
    "TRIM_ARRAY",
    "TRUE",
    "TRUNCATE",
    "UESCAPE",
    "UNBOUNDED",
    "UNCOMMITTED",
    "UNION",
    "UNIQUE",
    "UNKNOWN",
    "UNNEST",
    "UPDATE",
    "UPPER",
    "USER",
    "USING",
    "UUID",
    "VALUE",
    "VALUES",
    "VALUE_OF",
    "VARBINARY",
    "VARCHAR",
    "VARYING",
    "VAR_POP",
    "VAR_SAMP",
    "VERSIONING",
    "VIEW",
    "VIRTUAL",
    "WHEN",
    "WHENEVER",
    "WHERE",
    "WIDTH_BUCKET",
    "WINDOW",
    "WITH",
    "WITHIN",
    "WITHOUT",
    "WORK",
    "WRITE",
    "YEAR",
    "ZONE",
];

const SCHEMA_TABLE: &'static str = "information_schema";

pub async fn connect(profile: &Profile) -> Result<MySqlPool> {
    let conn = MySqlConnectOptions::new()
        .host(&profile.host)
        .port(profile.port)
        .ssl_mode(MySqlSslMode::Disabled);
    let conn = if let Some(ref user) = profile.user {
        conn.username(user)
    } else {
        conn
    };
    let conn = if let Some(ref pass) = profile.password {
        conn.password(pass)
    } else {
        conn
    };
    let conn = conn.database(&profile.db);
    let conn = if let Some(ref ssl_mode) = profile.ssl_mode {
        let mode = match ssl_mode {
            crate::config::SslMode::Disabled => MySqlSslMode::Disabled,
            crate::config::SslMode::Preferred => MySqlSslMode::Preferred,
            crate::config::SslMode::Required => MySqlSslMode::Required,
            crate::config::SslMode::VerifyCa => MySqlSslMode::VerifyCa,
            crate::config::SslMode::VerifyIdentity => MySqlSslMode::VerifyIdentity,
        };
        conn.ssl_mode(mode)
    } else {
        conn.ssl_mode(MySqlSslMode::Disabled)
    };
    let conn = if let Some(ref ca_file) = profile.ssl_ca {
        conn.ssl_ca(ca_file)
    } else {
        conn
    };
    Ok(MySqlPool::connect_with(conn)
        .await
        .with_context(|| crate::fl!("connect-failed"))?)
}

pub async fn all_databases(pool: &MySqlPool) -> Result<HashSet<String>> {
    let query: Vec<(String,)> = sqlx::query_as("SHOW DATABASES").fetch_all(pool).await?;
    let mut databases = HashSet::new();
    query.into_iter().for_each(|(db,)| {
        databases.insert(db);
    });
    Ok(databases)
}

pub async fn all_tables(pool: &MySqlPool) -> Result<HashSet<String>> {
    let query: Vec<(String,)> = sqlx::query_as("SHOW TABLES").fetch_all(pool).await?;
    let mut tables = HashSet::new();
    query.into_iter().for_each(|(t,)| {
        tables.insert(t);
    });
    Ok(tables)
}

pub async fn all_columns(
    pool: &MySqlPool,
    tables: &HashSet<String>,
) -> Result<HashMap<String, HashSet<String>>> {
    let mut columns: HashMap<String, HashSet<String>> = HashMap::new();
    let sql = format!(
        "SELECT TABLE_NAME, COLUMN_NAME FROM {}.COLUMNS WHERE table_name IN ({})",
        SCHEMA_TABLE,
        tables
            .iter()
            .map(|t| format!("'{}'", t))
            .collect::<Vec<String>>()
            .join(",")
    );
    let query: Vec<(String, String)> = sqlx::query_as(&sql).fetch_all(pool).await?;
    query.into_iter().for_each(|(table, col)| {
        if let Some(table) = columns.get_mut(&table) {
            table.insert(col);
        } else {
            let mut entry = HashSet::new();
            entry.insert(col);
            columns.insert(table, entry);
        }
    });
    Ok(columns)
}
