extern crate css_modules;

use css_modules::stylesheet::*;
use maplit::hashmap;

#[test]
fn parse_empty_stylesheet() {
    assert_eq!(
        Stylesheet::new("module", r#""#).unwrap(),
        Stylesheet {
            name: "module".to_string(),
            children: vec![],
            identifiers: hashmap![],
        }
    )
}

#[test]
fn parse_empty_ruleset() {
    let expected_class = "module_foobar_0";

    assert_eq!(
        Stylesheet::new("module", r#"p.foobar {}"#).unwrap(),
        Stylesheet {
            name: "module".to_string(),
            children: vec![Node::SelectRule {
                rule: Some(format!("p.{}", expected_class)),
                children: vec![]
            }],
            identifiers: hashmap![
                "foobar".to_string() => expected_class.to_string()
            ],
        }
    )
}

#[test]
fn parse_ruleset_with_property() {
    let expected_class = "module_foobar_0";

    assert_eq!(
        Stylesheet::new("module", r#"p.foobar { color: #fff; }"#).unwrap(),
        Stylesheet {
            name: "module".to_string(),
            children: vec![Node::SelectRule {
                rule: Some(format!("p.{}", expected_class)),
                children: vec![Node::Property {
                    name: Some("color".to_string()),
                    value: Some("#fff".to_string())
                }]
            }],
            identifiers: hashmap![
                "foobar".to_string() => expected_class.to_string()
            ],
        }
    )
}

#[test]
fn parse_empty_keyframes() {
    let expected_animation = "module_foobar_0";

    assert_eq!(
        Stylesheet::new("module", r#"@keyframes foobar {}"#).unwrap(),
        Stylesheet {
            name: "module".to_string(),
            children: vec![Node::AtRule {
                name: Some("keyframes".to_string()),
                rule: Some(format!("{} ", expected_animation)),
                children: vec![]
            }],
            identifiers: hashmap![
                "foobar".to_string() => expected_animation.to_string()
            ],
        }
    )
}

#[test]
fn parse_unclosed_block() {
    assert!(Stylesheet::new("module", r#"p {"#).is_err());
}

#[test]
fn format_empty_stylesheet() {
    let stylesheet = Stylesheet::new("module", r#""#).unwrap();

    assert_eq!(String::new(), format!("{}", stylesheet));
}

#[test]
fn format_empty_ruleset() {
    let stylesheet = Stylesheet::new("module", r#"p.foobar {}"#).unwrap();

    assert_eq!(String::new(), format!("{}", stylesheet));
}

#[test]
fn format_ruleset_with_property() {
    let stylesheet = Stylesheet::new("module", r#" p.foobar  {  color :  #fff ;  }"#).unwrap();
    let expected_class = "module_foobar_0";

    assert_eq!(
        format!("p.{} {{\n\tcolor: #fff;\n}}\n", expected_class),
        format!("{}", stylesheet)
    );
}

#[test]
fn format_empty_keyframes() {
    let stylesheet = Stylesheet::new("module", r#"@keyframes foobar {}"#).unwrap();
    let expected_animation = "module_foobar_0";

    assert_eq!(
        format!("@keyframes {};\n", expected_animation),
        format!("{}", stylesheet)
    );
}
