//! The [CSS Modules] project defines CSS Modules as:
//!
//! > A **CSS Module** is a CSS file in which all class names and animation names are scoped locally by default.
//!
//! This implementation is however currently immature and what parsing we do have is very naively implemented. As a result, currently only class names are locally scoped and the following work is in progress:
//!
//! - Locally scoped animation names
//! - Inlining `url()` and `@import` statements
//!
//! ## Usage
//!
//! A [`Module`] can be constructed manually:
//!
//! ```
//! use css_modules::Module;
//!
//! let css = Module::new("my_module", ".myStyles {}");
//! ```
//!
//! Or if you would prefer to use automatic module naming based on source code, through a macro:
//!
//! ```
//! use css_modules::*;
//!
//! let css = css_module!(".myStyles {}");
//! ```
//!
//! The same as above, but from a file relative to the current source file:
//!
//! ```
//! use css_modules::*;
//!
//! let css = include_css_module!("test.css");
//! ```
//!
//! Which is the equivelent of doing:
//!
//! ```
//! use css_modules::*;
//!
//! let css = css_module!(include_str!("test.css"));
//! ```
//!
//! [CSS Modules]: https://github.com/css-modules/css-modules
extern crate backtrace;
extern crate lazy_static;
extern crate regex;

use std::collections::HashMap;
use std::io::{Cursor, Error, Read, Seek, SeekFrom, Write};

/// A CSS Stylesheet that [`Module`]s can be written to
#[derive(Debug)]
pub struct Stylesheet {
    pub resource: Cursor<Vec<u8>>,
}

impl Default for Stylesheet {
    fn default() -> Self {
        Stylesheet {
            resource: Cursor::new(Vec::new()),
        }
    }
}

impl Stylesheet {
    pub fn new() -> Self {
        Self::default()
    }

    /// Write a module to the stylesheet.
    pub fn write(&mut self, module: &Module) -> Result<usize, Error> {
        self.resource.write(module.stylesheet.as_bytes())
    }

    pub fn to_string(&mut self) -> String {
        let mut result = String::new();

        self.resource.seek(SeekFrom::Start(0)).unwrap();

        self.resource
            .read_to_string(&mut result)
            .expect("Nothing to read.");

        result
    }
}

/// A CSS Module
#[derive(Clone, Debug, PartialEq)]
pub struct Module {
    pub name: String,
    pub stylesheet: String,
    pub classes: HashMap<String, String>,
}

impl Default for Module {
    fn default() -> Self {
        Module {
            name: "unknown_module".to_string(),
            stylesheet: String::new(),
            classes: HashMap::new(),
        }
    }
}

impl Module {
    /// Build a new [`Module`].
    pub fn new(name: &str, stylesheet: &str) -> Module {
        use lazy_static::lazy_static;
        use regex::{Captures, Regex};

        lazy_static! {
            static ref CLASSES: Regex = Regex::new(r"\.[\w\d\-_]+").unwrap();
        }

        let name = name.to_owned();
        let stylesheet = stylesheet.to_owned();
        let mut classes = HashMap::new();

        let stylesheet = CLASSES
            .replace_all(&stylesheet, |caps: &Captures| {
                let original = &caps[0][1..].to_owned();
                let original = original.to_string();
                let replacement = format!("{}_{}_{}", &name, &original, classes.len());
                let result = format!(".{}", replacement);

                classes.entry(original).or_insert(replacement);

                result
            })
            .to_string();

        Module {
            name,
            stylesheet,
            classes,
        }
    }

    /// Does the module contain a class?
    ///
    /// ```
    /// use css_modules::Module;
    ///
    /// let css = Module::new("my_module", ".myClass {}");
    ///
    /// assert!(css.contains("myClass"));
    /// ```
    pub fn contains(&self, class: &str) -> bool {
        self.classes.contains_key(&class.to_owned())
    }

    /// Get the alias of a class if it exists, otherwise return the one we were given.
    ///
    /// ```
    /// use css_modules::Module;
    ///
    /// let css = Module::new("my_module", ".myClass {}");
    ///
    /// assert_ne!("myClass", css.get("myClass"));
    /// assert_eq!("notMyClass", css.get("notMyClass"));
    /// ```
    pub fn get(&self, class: &str) -> String {
        if let Some(class) = self.classes.get(class) {
            class.to_owned()
        } else {
            class.to_owned()
        }
    }
}

/// Generate a css module name based on the location from where it was called.
#[macro_export]
macro_rules! css_module_name {
    () => {{
        use backtrace::{Backtrace, BacktraceFrame, BacktraceSymbol};
        use lazy_static::lazy_static;
        use regex::Regex;

        lazy_static! {
            static ref NAME_FORMATTER: Regex = Regex::new(r"[^\w]+").unwrap();
        }

        fn previous_symbol(level: u32) -> Option<BacktraceSymbol> {
            let (trace, curr_file, curr_line) = (Backtrace::new(), file!(), line!());
            let frames = trace.frames();

            frames
                .iter()
                .flat_map(BacktraceFrame::symbols)
                .skip_while(|s| {
                    s.filename()
                        .map(|p| !p.ends_with(curr_file))
                        .unwrap_or(true)
                        || s.lineno() != Some(curr_line)
                })
                .nth(1 + level as usize)
                .cloned()
        }

        let symbol = previous_symbol(0);
        let name = symbol.as_ref().and_then(BacktraceSymbol::name);

        if let Some(name) = name {
            NAME_FORMATTER
                .replace_all(&format!("{:?}", name), "_")
                .to_string()
        } else {
            "unknown_module".to_string()
        }
    }};
}

/// Make a [`Module`] from a string.
///
/// ```
/// use css_modules::*;
///
/// let css = css_module!(".myClass { font-weight: bold; color: red; }");
///
/// // Get the localised class name:
/// css.get("myClass");
///
/// // Get the module stylesheet as a string:
/// css.stylesheet;
/// ```
#[macro_export]
macro_rules! css_module {
    ($stylesheet:expr) => {{
        Module::new(&css_module_name!(), $stylesheet)
    }};
    ($name:expr, $stylesheet:expr) => {{
        Module::new($name, $stylesheet)
    }};
}

/// Make a [`Module`] from a file relative to the current source file.
///
/// ```
/// use css_modules::*;
///
/// let css = include_css_module!("test.css");
///
/// // Get the localised class name:
/// css.get("myClass");
///
/// // Get the module stylesheet as a string:
/// css.stylesheet;
/// ```
#[macro_export]
macro_rules! include_css_module {
    ($file:expr) => {{
        css_module!(include_str!($file))
    }};
    ($name:expr, $file:expr) => {{
        css_module!($name, include_str!($file))
    }};
}
