//! Configuration utilities for chisel modules.
//! The two main traits are FromYaml and FromArgs, for config-driven and oneliner mode
//! respectively.
//! This module implements the ChiselConfig structure, which contains the information for a number
//! of chisel executions.
//! A ChiselConfig contains a list of ruleset configurations and their names.
//! A ruleset configuration contains a set of top-level options and a set of module configurations to execute.
//! The aforementioned traits are used to generate ChiselConfigs.

use std::collections::HashMap;
use std::collections::VecDeque;
use std::fmt::{self, Display};

use serde_yaml::Value;

/// Main trait for generating a configuration usable by the driver from deserialized YAML.
pub trait FromYaml {
    // yaml: top-level yaml element generated by parser
    fn from_yaml(yaml: &Value) -> Result<Self, String>
    where
        Self: Sized;
}

/// Main trait for generating a configuration usable by the driver from CLI arguments.
pub trait FromArgs {
    // Supported formatting example:
    // modules: "remapimports, trimexports, verifyimports"
    // options: "remapimports.preset=ewasm, trimexport.option=foo, verifyimports.a=b"
    fn from_args(modules: &str, options: &str) -> Result<Self, String>
    where
        Self: Sized;
}

/// A list of rulesets for a single chisel run.
/// Left-hand value: name of ruleset
/// Right-hand value: ruleset configuration
#[derive(Debug)]
pub struct ChiselConfig(VecDeque<(String, Ruleset)>);

/// A single ruleset, containing options and a set of module configurations.
#[derive(Debug)]
pub struct Ruleset {
    /// Top-level ruleset configurations, such as the input/output file paths.
    options: HashMap<String, String>,
    /// List of module configurations.
    /// Left-hand value: name of module
    /// Right-hand value: module options
    modules: VecDeque<(String, ModuleConfig)>,
}

/// A key-value module configuration.
#[derive(Debug)]
pub struct ModuleConfig(HashMap<String, String>);

impl ChiselConfig {
    pub fn rulesets(&self) -> &VecDeque<(String, Ruleset)> {
        &self.0
    }

    pub fn rulesets_mut(&mut self) -> &mut VecDeque<(String, Ruleset)> {
        &mut self.0
    }
}

impl Ruleset {
    pub fn options(&self) -> &HashMap<String, String> {
        &self.options
    }

    pub fn options_mut(&mut self) -> &mut HashMap<String, String> {
        &mut self.options
    }

    pub fn modules(&self) -> &VecDeque<(String, ModuleConfig)> {
        &self.modules
    }

    pub fn modules_mut(&mut self) -> &mut VecDeque<(String, ModuleConfig)> {
        &mut self.modules
    }
}

impl ModuleConfig {
    pub fn options(&self) -> &HashMap<String, String> {
        &self.0
    }
}

impl FromArgs for ChiselConfig {
    fn from_args(modules: &str, options: &str) -> Result<Self, String>
    where
        Self: Sized,
    {
        let mut ret = VecDeque::new();
        match Ruleset::from_args(modules, options) {
            Ok(rs) => {
                ret.push_back(("cli".to_string(), rs));
                Ok(ChiselConfig(ret))
            }
            Err(e) => Err(e),
        }
    }
}

impl FromArgs for Ruleset {
    fn from_args(modules: &str, options: &str) -> Result<Self, String>
    where
        Self: Sized,
    {
        // Filter out whitespace for names.
        let modules: String = modules.chars().filter(|c| *c != ' ').collect();

        // Collect module names.
        let module_names: Vec<String> = modules
            .split(',')
            .filter_map(|c| match c {
                "" => None, // If the user put a comma somewhere weird, we need to filter out the empty string.
                any => Some(any.to_string()),
            })
            .collect();

        // Filter out whitespace for options.
        let options: String = options.chars().filter(|c| *c != ' ').collect();

        // Collect module options.
        let module_options: Vec<String> = options
            .split(',')
            .filter_map(|c| match c {
                "" => None,
                any => Some(any.to_string()),
            })
            .collect();

        let mut module_configs = VecDeque::new();
        // Iterate over each module and generate a config from its provided options.
        for name in module_names.iter() {
            // Iterate over the collected options and find the options corresponding to this
            // module.
            let relevant_options: Vec<String> = module_options
                .iter()
                .filter_map(|option| {
                    // Split by dot here to isolate the module name.
                    let option_pair: Vec<&str> = option.split('.').collect();
                    assert!(!option_pair.is_empty());

                    // If the module name is the same as the one we're getting options for, then
                    // yield the option.
                    if *name == option_pair[0] {
                        Some(option.to_string())
                    } else {
                        None
                    }
                })
                .collect();
            // Prepare the configuration map for this specific module.
            let mut options_final = HashMap::new();

            // Now we have all the relevant options, still formatted as "module.option=value"
            for option in relevant_options {
                // Get a string for all chars after the first dot, which should yield
                // "option=value"
                let keyval_pair: String =
                    option.chars().skip_while(|c| *c != '.').skip(1).collect();

                // Split into its terms.
                let keyval_pair: Vec<&str> = keyval_pair.split('=').collect();

                if keyval_pair.len() == 2 {
                    chisel_debug!(
                        1,
                        "Found option for module {}: {} = {}",
                        name,
                        &keyval_pair[0],
                        &keyval_pair[1]
                    );
                    options_final.insert(keyval_pair[0].to_string(), keyval_pair[1].to_string());
                } else {
                    return Err(format!(
                        "In module {}: option {} malformed: unexpected equals sign",
                        name, keyval_pair[0]
                    ));
                }
            }

            // Add the complete module config.
            module_configs.push_back((name.to_string(), ModuleConfig(options_final)));
        }

        // Since this is used in oneliner mode, the only useful "top-level" option is the input
        // file. We set this manually in oneliner mode, so we can leave this empty.
        let options: HashMap<String, String> = HashMap::new();
        Ok(Ruleset {
            options,
            modules: module_configs,
        })
    }
}

impl FromYaml for ChiselConfig {
    fn from_yaml(yaml: &Value) -> Result<Self, String>
    where
        Self: Sized,
    {
        // Ensure that the first value in the config is a mapping representing rulesets.
        if !yaml.is_mapping() {
            return Err("Top-level value is not a mapping".to_string());
        }

        let rulesets = yaml.as_mapping().expect("Already validated");

        // Ensure that all rulesets in the top-level YAML map are string-map pairs.
        // Valid example:
        // ruleset:
        //   option: value
        //
        // Invalid example:
        // ruleset: "something"
        if rulesets
            .iter()
            .any(|(key, val)| !key.is_string() || !val.is_mapping())
        {
            return Err(
                "Malformed ruleset; expected all rulesets to be string-map pairs".to_string(),
            );
        }

        let rulesets: VecDeque<(String, Ruleset)> = rulesets
            .iter()
            .map(|(key, val)| {
                (
                    key.as_str().expect("Already validated").to_string(),
                    Ruleset::from_yaml(val).expect("No failure cases yet"),
                )
            })
            .collect();

        Ok(ChiselConfig(rulesets))
    }
}

impl FromYaml for Ruleset {
    fn from_yaml(yaml: &Value) -> Result<Self, String>
    where
        Self: Sized,
    {
        let mut options: HashMap<String, String> = HashMap::new();
        let mut modules: VecDeque<(String, ModuleConfig)> = VecDeque::new();

        let ruleset = yaml.as_mapping().expect("Already validated");
        for (key, val) in ruleset.iter() {
            match (key, val) {
                // If string-string pair, treat it as an option.
                (Value::String(k), Value::String(v)) => {
                    options.insert(k.to_string(), v.to_string());
                }
                // If string-map pair or a string-null pair (no special options), treat it as a module configuration.
                (Value::String(k), v @ Value::Mapping(_)) | (Value::String(k), v @ Value::Null) => {
                    modules.push_back((
                        k.to_string(),
                        ModuleConfig::from_yaml(v).expect("No failure cases yet"),
                    ));
                }
                // ignore others for now
                (_, _) => (),
            }
        }
        Ok(Ruleset { options, modules })
    }
}

impl FromYaml for ModuleConfig {
    fn from_yaml(yaml: &Value) -> Result<Self, String>
    where
        Self: Sized,
    {
        if yaml == &Value::Null {
            Ok(ModuleConfig(HashMap::new()))
        } else {
            let options = yaml.as_mapping().expect("Already validated");
            let options: HashMap<String, String> = options
                .iter()
                .filter_map(|(key, val)| {
                    match (key, val) {
                        (Value::String(k), Value::String(v)) => {
                            Some((k.to_string(), v.to_string()))
                        }
                        (Value::String(k), Value::Bool(v)) => Some((
                            k.to_string(),
                            if *v {
                                "true".to_string()
                            } else {
                                "false".to_string()
                            },
                        )),
                        (Value::String(k), Value::Number(n)) => {
                            // Do not support floating point right now
                            if n.is_u64() || n.is_i64() {
                                Some((k.to_string(), n.to_string()))
                            } else {
                                None
                            }
                        }
                        (Value::String(k), Value::Sequence(v)) => {
                            // Ignore non-string/int/bools
                            // This will produce a comma-separated list of sequence elements.
                            Some((
                                k.to_string(),
                                v.iter()
                                    .filter_map(|elem| match elem {
                                        Value::String(s) => Some(s.to_string()),
                                        Value::Bool(v) => Some(if *v {
                                            "true".to_string()
                                        } else {
                                            "false".to_string()
                                        }),
                                        Value::Number(n) => {
                                            if n.is_u64() || n.is_i64() {
                                                Some(n.to_string())
                                            } else {
                                                None
                                            }
                                        }
                                        _ => None,
                                    })
                                    .fold(String::new(), |acc, elem| acc + &elem),
                            ))
                        }
                        _ => None,
                    }
                })
                .collect();

            Ok(ModuleConfig(options))
        }
    }
}

impl Display for ChiselConfig {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        // Fold into the last write error, if any
        self.0
            .iter()
            .map(|(name, ruleset)| write!(f, "\nRuleset {}:{}", name, ruleset))
            .fold(
                Ok(()),
                |acc, result| if result.is_err() { result } else { acc },
            )
    }
}

impl Display for Ruleset {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let options = self
            .options
            .iter()
            .map(|(key, val)| write!(f, "\n\t{}: {}", key, val))
            .fold(
                Ok(()),
                |acc, result| if result.is_err() { result } else { acc },
            );
        let modules = self
            .modules
            .iter()
            .map(|(name, module)| write!(f, "\n\tModule '{}':{}", name, module))
            .fold(
                Ok(()),
                |acc, result| if result.is_err() { result } else { acc },
            );

        if modules.is_err() {
            modules
        } else if options.is_err() {
            options
        } else {
            Ok(())
        }
    }
}

impl Display for ModuleConfig {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        self.0
            .iter()
            .map(|(key, val)| write!(f, "\n\t\t{}: {}", key, val))
            .fold(
                Ok(()),
                |acc, result| if result.is_err() { result } else { acc },
            )
    }
}

#[cfg(test)]
mod tests {
    use serde_yaml::Mapping;

    use super::*;

    #[test]
    fn from_yaml() {
        let yaml = {
            // test:
            //   preset: "ewasm"
            let mut test_module = Mapping::new();
            test_module.insert(
                Value::String("preset".to_string()),
                Value::String("ewasm".to_string()),
            );
            let test_module = Value::Mapping(test_module);

            // ruleset:
            //   file: "test.yaml"
            //   $test_module
            let mut test_config = Mapping::new();
            test_config.insert(
                Value::String("file".to_string()),
                Value::String("test.yaml".to_string()),
            );
            test_config.insert(Value::String("test".to_string()), test_module);
            let test_config = Value::Mapping(test_config);

            let mut ruleset = Mapping::new();
            ruleset.insert(Value::String("ruleset".to_string()), test_config);

            Value::Mapping(ruleset)
        };

        let config = ChiselConfig::from_yaml(&yaml).expect("Should not fail");

        assert_eq!(config.rulesets().len(), 1);
        assert_eq!(
            config.rulesets()[0].1.options().get("file"),
            Some(&"test.yaml".to_string())
        );
        assert_eq!(config.rulesets()[0].1.modules().len(), 1);
        assert_eq!(
            config.rulesets()[0].1.modules()[0]
                .1
                .options()
                .get(&"preset".to_string())
                .expect("Should be Some"),
            &"ewasm".to_string()
        );
    }

    #[test]
    fn from_args() {
        let config = ChiselConfig::from_args(
            "test,test1,test2",
            "test.preset=foo,test1.preset=bar,test2.wildcard=baz",
        )
        .expect("Should be valid");

        assert_eq!(config.rulesets().len(), 1);
        assert_eq!(config.rulesets()[0].1.modules().len(), 3);
        assert_eq!(
            config.rulesets()[0].1.modules()[0]
                .1
                .options()
                .get(&"preset".to_string())
                .expect("Should be Some"),
            &"foo".to_string()
        );
        assert_eq!(
            config.rulesets()[0].1.modules()[1]
                .1
                .options()
                .get(&"preset".to_string())
                .expect("Should be Some"),
            &"bar".to_string()
        );
        assert_eq!(
            config.rulesets()[0].1.modules()[2]
                .1
                .options()
                .get(&"wildcard".to_string())
                .expect("Should be Some"),
            &"baz".to_string()
        );
    }

    #[test]
    fn from_args_toomanyequals() {
        let config = ChiselConfig::from_args("test", "test.preset=foo=bar");

        assert!(config.is_err());
    }

    #[test]
    fn from_args_multidot() {
        let config =
            ChiselConfig::from_args("test", "test.option=value.subvalue").expect("Should be valid");
        assert_eq!(
            config.rulesets()[0].1.modules()[0]
                .1
                .options()
                .get(&"option".to_string())
                .expect("Should be Some"),
            &"value.subvalue".to_string()
        );

        let config =
            ChiselConfig::from_args("test", "test.option.suboption.subsuboption=value.subvalue")
                .expect("Should be valid");
        assert_eq!(
            config.rulesets()[0].1.modules()[0]
                .1
                .options()
                .get(&"option.suboption.subsuboption".to_string())
                .expect("Should be Some"),
            &"value.subvalue".to_string()
        );
    }

    #[test]
    fn from_args_nooptions() {
        // Should return a ruleset containing one module with no options.
        let config = ChiselConfig::from_args("test", "").expect("Should be valid");
        assert_eq!(config.rulesets()[0].1.modules()[0].1.options().len(), 0);
    }

    #[test]
    fn from_args_garbage_options() {
        // Should return a ruleset containing one module with no options.
        let config = ChiselConfig::from_args("test", "asdfkea;,aifr?akjw,akhtghdsje,")
            .expect("Should be valid");
        assert_eq!(config.rulesets()[0].1.modules()[0].1.options().len(), 0);

        // Should return an error.
        let config =
            ChiselConfig::from_args("test", "test.option.suboption.,subsuboption=value.subvalue");
        assert!(config.is_err());
    }

    #[test]
    fn yaml_not_mapping() {
        let yaml = Value::String("foobar".to_string());

        let config = ChiselConfig::from_yaml(&yaml);
        assert!(config.is_err());
    }

    #[test]
    fn yaml_mapping_type_mismatch() {
        // Key of incorrect type.
        let yaml = {
            let mut mapping = Mapping::new();
            mapping.insert(
                Value::Number(1337i64.into()),
                Value::Mapping(Mapping::new()),
            );
            Value::Mapping(mapping)
        };

        let config = ChiselConfig::from_yaml(&yaml);
        assert!(config.is_err());

        // Value of incorrect type.
        let yaml = {
            let mut mapping = Mapping::new();
            mapping.insert(
                Value::String("test".to_string()),
                Value::Number(12345i64.into()),
            );
            Value::Mapping(mapping)
        };

        let config = ChiselConfig::from_yaml(&yaml);
        assert!(config.is_err());
    }
}
