//! Config-driven chisel mode implementation.
//! The main entry point is chisel_run, which consumes the ChiselFlags generated by the caller.
//! chisel_run finds the configuration file provided and produces a ChiselConfig, which is passed
//! to the instantiated ChiselDriver and executed. Afterwards, results are written to the file
//! specified in the configuration.

use std::error::Error;
use std::fs::read_to_string;

use serde_yaml::Value;

use crate::config::ChiselConfig;
use crate::config::FromYaml;
use crate::driver::ChiselDriver;
use crate::driver::DriverState;
use crate::fail;
use crate::logger;
use crate::options::ChiselFlags;

/// Execute chisel in config-driven mode.
pub fn chisel_run(flags: ChiselFlags) -> i32 {
    let log_level = match flags.value_of("util.debugging") {
        Some("true") => 1i32,
        Some("false") => 0i32,
        _ => panic!("util.debugging must be set 'true' or 'false'"),
    };
    logger::set_global_log_level(log_level);

    chisel_debug!(1, "Running chisel in config-driven mode");
    chisel_debug!(1, "Flags set:");
    flags
        .iter()
        .for_each(|(k, v)| chisel_debug!(1, "\t{}: {}", k, v));

    let config_path = flags
        .value_of("run.config.path")
        .expect("'run.config.path' is not set!");

    chisel_debug!(1, "Resolving config at {}...", config_path);

    let path_resolved = match std::fs::canonicalize(config_path) {
        Ok(p) => p,
        Err(e) => fail(
            1,
            &format!(
                "could not resolve config path '{}': {}",
                config_path,
                e.description()
            ),
        ),
    };

    chisel_debug!(
        1,
        "Successfully resolved config path: {}",
        path_resolved.to_str().expect("Path is not valid unicode")
    );
    chisel_debug!(1, "Loading configuration...");

    let config = match read_to_string(path_resolved) {
        Ok(conf) => {
            chisel_debug!(1, "Successfully loaded configuration");
            conf
        }
        Err(e) => fail(
            1,
            &format!("failed to load configuration: {}", e.description()),
        ),
    };

    let yaml_parsed = serde_yaml::from_str::<Value>(&config).unwrap_or_else(|e| {
        fail(
            1,
            &format!("failed to parse configuration: {}", e.description()),
        )
    });

    // Validate basic properties of the YAML configuration.
    let chisel_config = match ChiselConfig::from_yaml(&yaml_parsed) {
        Ok(config) => {
            chisel_debug!(1, "Successfully processed configuration");
            config
        }
        Err(e) => fail(1, &format!("bad configuration: {}", e)),
    };

    let mut driver = ChiselDriver::new(chisel_config);

    loop {
        match driver.fire() {
            DriverState::Error(err, _) => {
                if flags.value_eq("util.norecover", "true") {
                    fail(1, &format!("runtime error: {}", err));
                } else {
                    // Forget the current ruleset and keep going
                    eprintln!("runtime error {}; skipping ruleset", err);
                    continue;
                }
            }
            DriverState::Done(_) => {
                break;
            }
            _ => panic!("Should never return READY"),
        }
    }

    // After execution, get results and write output.
    let mut results = driver.take_result();
    chisel_debug!(1, "Module execution completed successfully");
    eprintln!("{}", &results);
    results
        .rulesets_mut()
        .iter_mut()
        .map(|ruleset| {
            let ruleset_name = ruleset.name().to_string();
            let write_result = match flags.value_of("output.mode") {
                Some("bin") => ruleset.write("bin"),
                Some("wat") => ruleset.write("wat"),
                Some("hex") => ruleset.write("hex"),
                _ => panic!("CLI parser ensures value can only be one of the above"),
            };
            (ruleset_name, write_result)
        })
        .filter_map(|(name, result)| {
            if result.is_err() {
                Some((name, result.unwrap_err()))
            } else {
                if result.expect("Errors sorted out already") {
                    eprintln!("Ruleset {}: wrote output to file successfully.", name);
                } else {
                    eprintln!("Ruleset {}: no changes made to be written.", name);
                }
                None
            }
        })
        .for_each(|(name, error)| {
            eprintln!(
                "failed to write output from ruleset {} to file: {}",
                name,
                error.description()
            )
        });
    0
}
