use cargo_test_macro::cargo_test;
use cargo_test_support::{cargo_dir, main_file, project};
use std::env;

#[cargo_test]
fn patch_invalid_dependency() {
    let manifest = r#"
        [package]
        name = "example"
        version = "0.1.0"
        authors = ["wycats@example.com"]

        [dependencies]
        asdf = "1.0"

        [package.metadata.patch.asdf]
        patches = [
            "test.patch"
        ]
    "#;
    let p = project()
        .file("Cargo.toml", &manifest)
        .file("src/foo.rs", &main_file(r#""i am foo""#, &[]))
        .file("test.patch", r#""#)
        .build();

    let patch_bin =
        cargo_dir().join(format!("cargo-patch{}", env::consts::EXE_SUFFIX));
    p.process(&patch_bin)
        .with_stderr("Error: Unable to find package: asdf\n")
        .with_status(1)
        .run();
}

#[cargo_test]
fn patch_missing_patch() {
    let manifest = r#"
        [package]
        name = "example"
        version = "0.1.0"
        authors = ["wycats@example.com"]

        [dependencies]
        serde = "=1.0.110"

        [package.metadata.patch.serde]
        patches = [
            "test.patch"
        ]
    "#;
    let p = project()
        .file("Cargo.toml", &manifest)
        .file("src/foo.rs", &main_file(r#""i am foo""#, &[]))
        .build();

    let patch_bin =
        cargo_dir().join(format!("cargo-patch{}", env::consts::EXE_SUFFIX));
    p.process(&patch_bin)
        .with_stderr("Error: Unable to find patch file with path: \"test.patch\"\n")
        .with_status(1)
        .run();
}

#[cargo_test]
fn patch_invalid_patch() {
    let manifest = r#"
        [package]
        name = "example"
        version = "0.1.0"
        authors = ["wycats@example.com"]

        [dependencies]
        serde = "=1.0.110"

        [package.metadata.patch.serde]
        patches = [
            "test.patch"
        ]
    "#;
    let p = project()
        .file("Cargo.toml", &manifest)
        .file("src/foo.rs", &main_file(r#""i am foo""#, &[]))
        .file("test.patch", r#""#)
        .build();

    let patch_bin =
        cargo_dir().join(format!("cargo-patch{}", env::consts::EXE_SUFFIX));
    p.process(&patch_bin)
        .with_stderr("Error: Unable to parse patch file\n")
        .with_status(1)
        .run();
}

#[cargo_test]
fn patch_simple() {
    let manifest = r#"
        [package]
        name = "example"
        version = "0.1.0"
        authors = ["wycats@example.com"]

        [dependencies]
        serde = "=1.0.110"

        [package.metadata.patch.serde]
        patches = [
            "test.patch"
        ]
    "#;
    let patch = r#"--- LICENSE-MIT	2020-05-20 18:44:09.709027472 +0200
+++ "LICENSE-MIT copy"	2020-05-20 18:58:46.253762666 +0200
@@ -8,9 +8,7 @@
 is furnished to do so, subject to the following
 conditions:
 
-The above copyright notice and this permission notice
-shall be included in all copies or substantial portions
-of the Software.
+PATCHED
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF
 ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
"#;
    let p = project()
        .file("Cargo.toml", &manifest)
        .file("src/foo.rs", &main_file(r#""i am foo""#, &[]))
        .file("test.patch", &patch)
        .build();

    let patch_bin =
        cargo_dir().join(format!("cargo-patch{}", env::consts::EXE_SUFFIX));
    p.process(&patch_bin).with_stdout("Patched serde\n").run();

    let license_mit = p
        .build_dir()
        .join("patch")
        .join("serde-1.0.110")
        .join("LICENSE-MIT");
    let licenes =
        std::fs::read_to_string(license_mit).expect("Unable to read license file");
    assert!(licenes.contains("PATCHED"));
}

#[cargo_test]
fn patch_detailed() {
    let manifest = r#"
        [package]
        name = "example"
        version = "0.1.0"
        authors = ["wycats@example.com"]

        [dependencies]
        serde = { version = "=1.0.110", features = ["derive"] }

        [package.metadata.patch.serde]
        patches = [
            "test.patch"
        ]
    "#;
    let patch = r#"--- LICENSE-MIT	2020-05-20 18:44:09.709027472 +0200
+++ "LICENSE-MIT copy"	2020-05-20 18:58:46.253762666 +0200
@@ -8,9 +8,7 @@
 is furnished to do so, subject to the following
 conditions:
 
-The above copyright notice and this permission notice
-shall be included in all copies or substantial portions
-of the Software.
+PATCHED
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF
 ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
"#;
    let p = project()
        .file("Cargo.toml", &manifest)
        .file("src/foo.rs", &main_file(r#""i am foo""#, &[]))
        .file("test.patch", &patch)
        .build();

    let patch_bin =
        cargo_dir().join(format!("cargo-patch{}", env::consts::EXE_SUFFIX));
    p.process(&patch_bin).with_stdout("Patched serde\n").run();

    let license_mit = p
        .build_dir()
        .join("patch")
        .join("serde-1.0.110")
        .join("LICENSE-MIT");
    let licenes =
        std::fs::read_to_string(license_mit).expect("Unable to read license file");
    assert!(licenes.contains("PATCHED"));
}
