extern crate basic_dsp;
use basic_dsp::*;
use basic_dsp::conv_types::*;

use std::fs::File;
use std::io::prelude::*;
use std::io;
use std::f64::consts::PI;

fn main() {
    let number_of_symbols = 10000;
    let mut prbs = Prbs15::new();
    let mut channel1 = RealTimeVector64::from_constant(0.0, number_of_symbols);
    let mut channel2 = RealTimeVector64::from_constant(0.0, number_of_symbols);
    let mut empty = ComplexTimeVector64::empty();
    
    for i in 0..3 {
        fill_vectors_with_prbs(&mut channel1, &mut channel2, &mut prbs);
        let complex = 
            empty.set_real_imag(&channel1, &channel2)
            .and_then(|v| v.interpolatei(&RaisedCosineFunction::new(0.35), 10))
            .expect("Complex baseband vector from real vectors");
        let mut file = File::create(format!("baseband_time{}.csv", i)).expect("Failed to create baseband time file");
        complex_vector_to_file(&complex, &mut file).expect("Failed to write baseband time file");
        
        let modulated = 
            complex.multiply_complex_exponential(0.25 * PI, 0.0)
            .expect("frequency shift");
        
        let real = 
            modulated.to_real()
            .expect("Mod to real");
            
        let mut file = File::create(format!("modulated_time{}.csv", i)).expect("Failed to create modulated time file");
        real_vector_to_file(&real, &mut file).expect("Failed to write modulated time file");
        
        let delta = real.delta();
        empty = real.rededicate_as_complex_time_vector(delta); // Reuse memory
    }
}

struct Prbs15 {
    lfsr: u32
}

impl Prbs15 {
    fn new() -> Self {
        Prbs15 { lfsr: 0x1 }
    }
    
    fn next(&mut self) -> f64 {
        let bit = (self.lfsr ^ self.lfsr >> 14) & 0x1;
        self.lfsr = (self.lfsr >> 1) | (bit << 14);
        (bit as f64 - 0.5)
    }
}

fn fill_vectors_with_prbs(
    channel1: &mut RealTimeVector64, 
    channel2: &mut RealTimeVector64, 
    prbs: &mut Prbs15) {
    assert!(channel1.points() == channel2.points());
    for i in 0..channel1.points() {
        channel2[i] = prbs.next();
        channel1[i] = prbs.next();
    }
}

fn complex_vector_to_file(vector: &ComplexTimeVector64, f: &mut File) -> io::Result <()>  {
    let mut i = 0;
    while i < vector.len() {
        try! { writeln!(f, "{}, {}", vector[i], vector[i + 1]) };
        i += 2;
    }
    Ok(())
}

fn real_vector_to_file(vector: &RealTimeVector64, f: &mut File) -> io::Result <()>  {
    let mut i = 0;
    while i < vector.len() {
        try! { writeln!(f, "{}", vector[i]) };
        i += 2;
    }
    Ok(())
}