use crate::custom_serde::*;
use chrono::{DateTime, Utc};
use serde::de::DeserializeOwned;
use serde::ser::Serialize;
use serde_json::Value;
use std::collections::HashMap;

/// `AutoScalingEvent` struct is used to parse the json for auto scaling event types //
#[derive(Debug, Clone, PartialEq, Deserialize, Serialize)]
#[serde(rename_all = "camelCase")]
pub struct AutoScalingEvent<T1 = Value>
where
    T1: DeserializeOwned,
    T1: Serialize,
{
    /// The version of event data
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    pub version: Option<String>,
    /// The unique ID of the event
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    pub id: Option<String>,
    /// Details about event type
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    #[serde(rename = "detail-type")]
    pub detail_type: Option<String>,
    /// Source of the event
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    pub source: Option<String>,
    /// AccountId
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    #[serde(rename = "account")]
    pub account_id: Option<String>,
    /// Event timestamp
    pub time: DateTime<Utc>,
    /// Region of event
    #[serde(deserialize_with = "deserialize_lambda_string")]
    #[serde(default)]
    pub region: Option<String>,
    /// Information about resources impacted by event
    pub resources: Vec<String>,
    #[serde(deserialize_with = "deserialize_lambda_map")]
    #[serde(default)]
    #[serde(bound = "")]
    pub detail: HashMap<String, T1>,
}

#[cfg(test)]
mod test {
    use super::*;

    extern crate serde_json;

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_launch_successful() {
        let data = include_bytes!("fixtures/example-autoscaling-event-launch-successful.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_launch_unsuccessful() {
        let data = include_bytes!("fixtures/example-autoscaling-event-launch-unsuccessful.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_lifecycle_action() {
        let data = include_bytes!("fixtures/example-autoscaling-event-lifecycle-action.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_terminate_action() {
        let data = include_bytes!("fixtures/example-autoscaling-event-terminate-action.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_terminate_successful() {
        let data = include_bytes!("fixtures/example-autoscaling-event-terminate-successful.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }

    #[test]
    #[cfg(feature = "autoscaling")]
    fn example_autoscaling_event_terminate_unsuccessful() {
        let data = include_bytes!("fixtures/example-autoscaling-event-terminate-unsuccessful.json");
        let parsed: AutoScalingEvent = serde_json::from_slice(data).unwrap();
        let output: String = serde_json::to_string(&parsed).unwrap();
        let reparsed: AutoScalingEvent = serde_json::from_slice(output.as_bytes()).unwrap();
        assert_eq!(parsed, reparsed);
    }
}
