from . import AWSObject, AWSProperty
from .validators import (
    boolean, double, integer_range, json_checker, positive_integer
)


def validate_integration_type(integration_type):

    valid_integration_types = [
        "AWS",
        "AWS_PROXY",
        "HTTP",
        "HTTP_PROXY",
        "MOCK"
    ]
    if integration_type not in valid_integration_types:
        raise ValueError(
            "{} is not a valid IntegrationType".format(
                integration_type)
        )
    return integration_type


def validate_authorizer_type(authorizer_type):

    valid_authorizer_types = [
        "REQUEST"
    ]
    if authorizer_type not in valid_authorizer_types:
        raise ValueError(
            "{} is not a valid AuthorizerType".format(
                authorizer_type)
        )
    return authorizer_type


def validate_logging_level(logging_level):

    valid_logging_levels = [
        "WARN",
        "INFO",
        "DEBUG"
    ]
    if logging_level not in valid_logging_levels:
        raise ValueError(
            "{} is not a valid LoggingLevel".format(
                logging_level)
        )
    return logging_level


def validate_passthrough_behavior(passthrough_behavior):

    valid_passthrough_behaviors = [
        "WHEN_NO_MATCH",
        "WHEN_NO_TEMPLATES",
        "NEVER"
    ]
    if passthrough_behavior not in valid_passthrough_behaviors:
        raise ValueError(
            "{} is not a valid PassthroughBehavior".format(
                passthrough_behavior)
        )
    return passthrough_behavior


def validate_content_handling_strategy(content_handling_strategy):

    valid_handling_strategy_values = [
        "CONVERT_TO_TEXT",
        "CONVERT_TO_BINARY"
    ]
    if content_handling_strategy not in valid_handling_strategy_values:
        raise ValueError(
            "{} is not a valid ContentHandlingStrategy".format(
                content_handling_strategy)
        )
    return content_handling_strategy


def validate_authorizer_ttl(ttl_value):
    """ Validate authorizer ttl timeout
    :param ttl_value: The TTL timeout in seconds
    :return: The provided TTL value if valid
    """
    ttl_value = int(positive_integer(ttl_value))
    if ttl_value > 3600:
        raise ValueError("The AuthorizerResultTtlInSeconds should be <= 3600")
    return ttl_value


class AccessLogSettings(AWSProperty):
    props = {
        "DestinationArn": (str, False),
        "Format": (str, False)
    }


class RouteSettings(AWSProperty):
    props = {
        "DataTraceEnabled": (str, False),
        "DetailedMetricsEnabled": (boolean, False),
        "LoggingLevel": (validate_logging_level, False),
        "ThrottlingBurstLimit": (positive_integer, False),
        "ThrottlingRateLimit": (double, False)
    }


class Api(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Api"

    props = {
        "ApiKeySelectionExpression": (str, False),
        "Description": (str, False),
        "DisableSchemaValidation": (boolean, False),
        "Name": (str, True),
        "ProtocolType": (str, True),
        "RouteSelectionExpression": (str, True),
        "Version": (str, False)
    }


class Authorizer(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Authorizer"

    props = {
        "ApiId":  (str, True),
        "AuthorizerCredentialsArn":  (str, False),
        "AuthorizerResultTtlInSeconds": (validate_authorizer_ttl, False),
        "AuthorizerType":  (validate_authorizer_type, True),
        "AuthorizerUri":  (str, True),
        "IdentitySource": ([str], True),
        "IdentityValidationExpression":  (str, False),
        "Name":  (str, True)
    }


class Deployment(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Deployment"

    props = {
        "ApiId": (str, True),
        "Description": (str, False),
        "StageName": (str, False),
    }


class Integration(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Integration"

    props = {
        "ApiId": (str, True),
        "ConnectionType": (str, False),
        "ContentHandlingStrategy": (validate_content_handling_strategy, False),
        "CredentialsArn": (str, False),
        "Description": (str, False),
        "IntegrationMethod": (str, False),
        "IntegrationType": (validate_integration_type, True),
        "IntegrationUri": (str, False),
        "PassthroughBehavior": (validate_passthrough_behavior, False),
        "RequestParameters": (dict, False),
        "RequestTemplates": (dict, False),
        "TemplateSelectionExpression": (str, False),
        "TimeoutInMillis": (integer_range(50, 29000), False)
    }


class IntegrationResponse(AWSObject):
    resource_type = "AWS::ApiGatewayV2::IntegrationResponse"

    props = {
        "ApiId": (str, True),
        "ContentHandlingStrategy": (validate_content_handling_strategy, False),
        "IntegrationId": (str, True),
        "IntegrationResponseKey": (str, True),
        "ResponseParameters": (dict, False),
        "ResponseTemplates": (dict, False),
        "TemplateSelectionExpression": (str, False)
    }


class Model(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Model"

    props = {
        "ApiId": (str, True),
        "ContentType": (str, False),
        "Description": (str, False),
        "Name": (str, True),
        "Schema": ((str, dict), True)
    }

    def validate(self):
        name = 'Schema'
        if name in self.properties:
            schema = self.properties.get(name)
            self.properties[name] = json_checker(schema)


class Route(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Route"

    props = {
        "ApiId": (str, True),
        "ApiKeyRequired": (boolean, False),
        "AuthorizationScopes": ([str], False),
        "AuthorizationType": (str, False),
        "AuthorizerId": (str, False),
        "ModelSelectionExpression": (str, False),
        "OperationName": (str, False),
        "RequestModels": (dict, False),
        "RequestParameters": (dict, False),
        "RouteKey": (str, True),
        "RouteResponseSelectionExpression": (str, False),
        "Target": (str, False)
    }


class RouteResponse(AWSObject):
    resource_type = "AWS::ApiGatewayV2::RouteResponse"

    props = {
        "ApiId": (str, True),
        "ModelSelectionExpression": (str, False),
        "ResponseModels": (dict, False),
        "ResponseParameters": (dict, False),
        "RouteId": (str, True),
        "RouteResponseKey": (str, True)
    }


class Stage(AWSObject):
    resource_type = "AWS::ApiGatewayV2::Stage"

    props = {
        "AccessLogSettings": (AccessLogSettings, False),
        "ApiId": (str, True),
        "ClientCertificateId": (str, False),
        "DefaultRouteSettings": (RouteSettings, False),
        "DeploymentId": (str, False),
        "Description": (str, False),
        "RouteSettings": (dict, False),
        "StageName": (str, True),
        "StageVariables": (dict, False)
    }
